context("SQL: escaping")

# Identifiers ------------------------------------------------------------------

ei <- function(...) unclass(escape(ident(c(...))))

test_that("identifiers are doubled quoted", {
  expect_equal(ei("x"), '"x"')
})

test_that("identifiers are comma separated", {
  expect_equal(ei("x", "y"), '"x", "y"')
})

test_that("identifier names become AS", {
  expect_equal(ei(x = "y"), '"y" AS "x"')
})


# Zero-length inputs ------------------------------------------------------

test_that("zero length inputs yield zero length output when not collapsed", {
  expect_equal(sql_vector(sql(), collapse = NULL), sql())
  expect_equal(sql_vector(ident(), collapse = NULL), sql())
  expect_equal(sql_vector(ident_q(), collapse = NULL), sql())
})

test_that("zero length inputs yield length-1 output when collapsed", {
  expect_equal(sql_vector(sql(), parens = FALSE, collapse = ""), sql(""))
  expect_equal(sql_vector(sql(), parens = TRUE, collapse = ""), sql("()"))
  expect_equal(sql_vector(ident(), parens = FALSE, collapse = ""), sql(""))
  expect_equal(sql_vector(ident(), parens = TRUE, collapse = ""), sql("()"))
  expect_equal(sql_vector(ident_q(), parens = FALSE, collapse = ""), sql(""))
  expect_equal(sql_vector(ident_q(), parens = TRUE, collapse = ""), sql("()"))
})

# Special values ----------------------------------------------------------------

test_that("missing vaues become null", {
  expect_equal(escape(NA), sql("NULL"))
  expect_equal(escape(NA_real_), sql("NULL"))
  expect_equal(escape(NA_integer_), sql("NULL"))
  expect_equal(escape(NA_character_), sql("NULL"))
})

test_that("-Inf and Inf are expanded and quoted", {
  expect_equal(escape(Inf), sql("'Infinity'"))
  expect_equal(escape(-Inf), sql("'-Infinity'"))
})

test_that("logical is SQL-99 compatible (by default)", {
  expect_equal(escape(TRUE), sql("TRUE"))
  expect_equal(escape(FALSE), sql("FALSE"))
  expect_equal(escape(NA), sql("NULL"))
})

test_that("can escape integer64 values", {
  skip_if_not_installed("bit64")

  expect_equal(escape(bit64::as.integer64(NA)), sql("NULL"))
  expect_equal(escape(bit64::as.integer64("123456789123456789")), sql("123456789123456789"))
})

# Times -------------------------------------------------------------------

test_that("times are converted to ISO 8601", {
  x <- ISOdatetime(2000, 1, 2, 3, 4, 5, tz = "US/Central")
  expect_equal(escape(x), sql("'2000-01-02T09:04:05Z'"))
})

# Name aliasing -----------------------------------------------------------

test_that("queries generated by select() don't alias unnecessarily", {
  lf_build <- lazy_frame(x = 1) %>%
    select(x) %>%
    sql_build()

  lf_render <- sql_render(lf_build, con = simulate_dbi())

  expect_equal(lf_render,  sql("SELECT \"x\"\nFROM \"df\""))
})

test_that("names_to_as() doesn't alias when ident name and value are identical", {
  x <- ident(name = "name")
  y <- sql_escape_ident(con = simulate_dbi(),  x = x)

  expect_equal(names_to_as(y, names2(x),  con = simulate_dbi()),  "\"name\"")
})

test_that("names_to_as() doesn't alias when ident name is missing", {
  x <- ident("*")
  y <- sql_escape_ident(con = simulate_dbi(),  x = x)

  expect_equal(names_to_as(y, names2(x),  con = simulate_dbi()),  "\"*\"")
})

test_that("names_to_as() aliases when ident name and value are different", {
  x <- ident(new_name = "name")
  y <- sql_escape_ident(con = simulate_dbi(),  x = x)

  expect_equal(names_to_as(y, names2(x),  con = simulate_dbi()),  "\"name\" AS \"new_name\"")
})
