static const char* op_c_source =
"/* This file is an image processing operation for GEGL                        \n"
" *                                                                            \n"
" * GEGL is free software; you can redistribute it and/or                      \n"
" * modify it under the terms of the GNU Lesser General Public                 \n"
" * License as published by the Free Software Foundation; either               \n"
" * version 3 of the License, or (at your option) any later version.           \n"
" *                                                                            \n"
" * GEGL is distributed in the hope that it will be useful,                    \n"
" * but WITHOUT ANY WARRANTY; without even the implied warranty of             \n"
" * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU          \n"
" * Lesser General Public License for more details.                            \n"
" *                                                                            \n"
" * You should have received a copy of the GNU Lesser General Public           \n"
" * License along with GEGL; if not, see <http://www.gnu.org/licenses/>.       \n"
" *                                                                            \n"
" * Copyright 2012 Victor Oliveira <victormatheus@gmail.com>                   \n"
" */                                                                           \n"
"                                                                              \n"
" /* This is an implementation of a fast approximated bilateral filter         \n"
"  * algorithm descripted in:                                                  \n"
"  *                                                                           \n"
"  *  A Fast Approximation of the Bilateral Filter using a Signal Processing Approach\n"
"  *  Sylvain Paris and Frédo Durand                                          \n"
"  *  European Conference on Computer Vision (ECCV'06)                         \n"
"  */                                                                          \n"
"                                                                              \n"
"#include \"config.h\"                                                         \n"
"#include <glib/gi18n-lib.h>                                                   \n"
"                                                                              \n"
"                                                                              \n"
"#ifdef GEGL_PROPERTIES                                                        \n"
"                                                                              \n"
"property_double (r_sigma, _(\"Smoothness\"), 50)                              \n"
"    description(_(\"Level of smoothness\"))                                   \n"
"    value_range (1, 100)                                                      \n"
"                                                                              \n"
"property_int (s_sigma, _(\"Blur radius\"), 8)                                 \n"
"   description(_(\"Radius of square pixel region, (width and height will be radius*2+1).\"))\n"
"   value_range (1, 100)                                                       \n"
"                                                                              \n"
"#else                                                                         \n"
"                                                                              \n"
"#define GEGL_OP_FILTER                                                        \n"
"#define GEGL_OP_C_SOURCE bilateral-filter-fast.c                              \n"
"                                                                              \n"
"#include \"gegl-op.h\"                                                        \n"
"#include <math.h>                                                             \n"
"                                                                              \n"
"inline static float lerp(float a, float b, float v)                           \n"
"{                                                                             \n"
"  return (1.0f - v) * a + v * b;                                              \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"bilateral_filter (GeglBuffer          *src,                                   \n"
"                  const GeglRectangle *src_rect,                              \n"
"                  GeglBuffer          *dst,                                   \n"
"                  const GeglRectangle *dst_rect,                              \n"
"                  gint                 s_sigma,                               \n"
"                  gfloat               r_sigma);                              \n"
"                                                                              \n"
"static gboolean                                                               \n"
"bilateral_cl_process (GeglOperation       *operation,                         \n"
"                      GeglBuffer          *input,                             \n"
"                      GeglBuffer          *output,                            \n"
"                      const GeglRectangle *result,                            \n"
"                      gint                 s_sigma,                           \n"
"                      gfloat               r_sigma);                          \n"
"                                                                              \n"
"#include <stdio.h>                                                            \n"
"                                                                              \n"
"static void bilateral_prepare (GeglOperation *operation)                      \n"
"{                                                                             \n"
"  gegl_operation_set_format (operation, \"input\",  babl_format (\"RGBA float\"));\n"
"  gegl_operation_set_format (operation, \"output\", babl_format (\"RGBA float\"));\n"
"}                                                                             \n"
"                                                                              \n"
"static GeglRectangle                                                          \n"
"bilateral_get_required_for_output (GeglOperation       *operation,            \n"
"                                  const gchar         *input_pad,             \n"
"                                  const GeglRectangle *roi)                   \n"
"{                                                                             \n"
"  GeglRectangle result = *gegl_operation_source_get_bounding_box (operation,  \n"
"                                                                  \"input\"); \n"
"  return result;                                                              \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static GeglRectangle                                                          \n"
"bilateral_get_cached_region (GeglOperation       *operation,                  \n"
"                            const GeglRectangle *roi)                         \n"
"{                                                                             \n"
"  return *gegl_operation_source_get_bounding_box (operation, \"input\");      \n"
"}                                                                             \n"
"                                                                              \n"
"static gboolean                                                               \n"
"bilateral_process (GeglOperation       *operation,                            \n"
"                   GeglBuffer          *input,                                \n"
"                   GeglBuffer          *output,                               \n"
"                   const GeglRectangle *result,                               \n"
"                   gint                 level)                                \n"
"{                                                                             \n"
"  GeglProperties   *o = GEGL_PROPERTIES (operation);                          \n"
"                                                                              \n"
"#if 0                                                                         \n"
"  if (gegl_operation_use_opencl (operation))                                  \n"
"    if (bilateral_cl_process (operation, input, output, result, o->s_sigma, o->r_sigma/100))\n"
"      return TRUE;                                                            \n"
"#endif                                                                        \n"
"                                                                              \n"
"  bilateral_filter (input, result, output, result, o->s_sigma, o->r_sigma/100);\n"
"                                                                              \n"
"  return  TRUE;                                                               \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"bilateral_filter (GeglBuffer          *src,                                   \n"
"                  const GeglRectangle *src_rect,                              \n"
"                  GeglBuffer          *dst,                                   \n"
"                  const GeglRectangle *dst_rect,                              \n"
"                  gint                 s_sigma,                               \n"
"                  gfloat               r_sigma)                               \n"
"{                                                                             \n"
"  gint c, x, y, z, k, i;                                                      \n"
"                                                                              \n"
"  const gint padding_xy = 2;                                                  \n"
"  const gint padding_z  = 2;                                                  \n"
"                                                                              \n"
"  const gint width    = src_rect->width;                                      \n"
"  const gint height   = src_rect->height;                                     \n"
"  const gint channels = 4;                                                    \n"
"                                                                              \n"
"  const gint sw = (width -1) / s_sigma + 1 + (2 * padding_xy);                \n"
"  const gint sh = (height-1) / s_sigma + 1 + (2 * padding_xy);                \n"
"  const gint depth = (int)(1.0f / r_sigma) + 1 + (2 * padding_z);             \n"
"                                                                              \n"
"  /* down-sampling */                                                         \n"
"                                                                              \n"
"  gfloat *grid     = g_new0 (gfloat, sw * sh * depth * channels * 2);         \n"
"  gfloat *blurx    = g_new0 (gfloat, sw * sh * depth * channels * 2);         \n"
"  gfloat *blury    = g_new0 (gfloat, sw * sh * depth * channels * 2);         \n"
"  gfloat *blurz    = g_new0 (gfloat, sw * sh * depth * channels * 2);         \n"
"                                                                              \n"
"  gfloat *input    = g_new0 (gfloat, width * height * channels);              \n"
"  gfloat *smoothed = g_new0 (gfloat, width * height * channels);              \n"
"                                                                              \n"
"  #define INPUT(x,y,c) input[c+channels*(x + width * y)]                      \n"
"                                                                              \n"
"  #define  GRID(x,y,z,c,i) grid [i+2*(c+channels*(x+sw*(y+z*sh)))]            \n"
"  #define BLURX(x,y,z,c,i) blurx[i+2*(c+channels*(x+sw*(y+z*sh)))]            \n"
"  #define BLURY(x,y,z,c,i) blury[i+2*(c+channels*(x+sw*(y+z*sh)))]            \n"
"  #define BLURZ(x,y,z,c,i) blurz[i+2*(c+channels*(x+sw*(y+z*sh)))]            \n"
"                                                                              \n"
"  gegl_buffer_get (src, src_rect, 1.0, babl_format (\"RGBA float\"), input, GEGL_AUTO_ROWSTRIDE,\n"
"                   GEGL_ABYSS_NONE);                                          \n"
"                                                                              \n"
"  for (k=0; k < (sw * sh * depth * channels * 2); k++)                        \n"
"    {                                                                         \n"
"      grid [k] = 0.0f;                                                        \n"
"      blurx[k] = 0.0f;                                                        \n"
"      blury[k] = 0.0f;                                                        \n"
"      blurz[k] = 0.0f;                                                        \n"
"    }                                                                         \n"
"                                                                              \n"
"#if 0                                                                         \n"
"  /* in case we want to normalize the color space */                          \n"
"                                                                              \n"
"  gfloat input_min[4] = { FLT_MAX,  FLT_MAX,  FLT_MAX};                       \n"
"  gfloat input_max[4] = {-FLT_MAX, -FLT_MAX, -FLT_MAX};                       \n"
"                                                                              \n"
"  for(y = 0; y < height; y++)                                                 \n"
"    for(x = 0; x < width; x++)                                                \n"
"      for(c = 0; c < channels; c++)                                           \n"
"        {                                                                     \n"
"          input_min[c] = MIN(input_min[c], INPUT(x,y,c));                     \n"
"          input_max[c] = MAX(input_max[c], INPUT(x,y,c));                     \n"
"        }                                                                     \n"
"#endif                                                                        \n"
"                                                                              \n"
"  /* downsampling */                                                          \n"
"                                                                              \n"
"  for(y = 0; y < height; y++)                                                 \n"
"    for(x = 0; x < width; x++)                                                \n"
"      for(c = 0; c < channels; c++)                                           \n"
"        {                                                                     \n"
"          const float z = INPUT(x,y,c); // - input_min[c];                    \n"
"                                                                              \n"
"          const int small_x = (int)((float)(x) / s_sigma + 0.5f) + padding_xy;\n"
"          const int small_y = (int)((float)(y) / s_sigma + 0.5f) + padding_xy;\n"
"          const int small_z = (int)((float)(z) / r_sigma + 0.5f) + padding_z; \n"
"                                                                              \n"
"          g_assert(small_x >= 0 && small_x < sw);                             \n"
"          g_assert(small_y >= 0 && small_y < sh);                             \n"
"          g_assert(small_z >= 0 && small_z < depth);                          \n"
"                                                                              \n"
"          GRID(small_x, small_y, small_z, c, 0) += INPUT(x, y, c);            \n"
"          GRID(small_x, small_y, small_z, c, 1) += 1.0f;                      \n"
"        }                                                                     \n"
"                                                                              \n"
"  /* blur in x, y and z */                                                    \n"
"  /* XXX: we could use less memory, but at expense of code readability */     \n"
"                                                                              \n"
"  for (z = 1; z < depth-1; z++)                                               \n"
"    for (y = 1; y < sh-1; y++)                                                \n"
"      for (x = 1; x < sw-1; x++)                                              \n"
"        for(c = 0; c < channels; c++)                                         \n"
"          for (i=0; i<2; i++)                                                 \n"
"            BLURX(x, y, z, c, i) = (GRID (x-1, y, z, c, i) + 2.0f * GRID (x, y, z, c, i) + GRID (x+1, y, z, c, i)) / 4.0f;\n"
"                                                                              \n"
"  for (z = 1; z < depth-1; z++)                                               \n"
"    for (y = 1; y < sh-1; y++)                                                \n"
"      for (x = 1; x < sw-1; x++)                                              \n"
"        for(c = 0; c < channels; c++)                                         \n"
"          for (i=0; i<2; i++)                                                 \n"
"            BLURY(x, y, z, c, i) = (BLURX (x, y-1, z, c, i) + 2.0f * BLURX (x, y, z, c, i) + BLURX (x, y+1, z, c, i)) / 4.0f;\n"
"                                                                              \n"
"  for (z = 1; z < depth-1; z++)                                               \n"
"    for (y = 1; y < sh-1; y++)                                                \n"
"      for (x = 1; x < sw-1; x++)                                              \n"
"        for(c = 0; c < channels; c++)                                         \n"
"          for (i=0; i<2; i++)                                                 \n"
"            BLURZ(x, y, z, c, i) = (BLURY (x, y-1, z, c, i) + 2.0f * BLURY (x, y, z, c, i) + BLURY (x, y+1, z, c, i)) / 4.0f;\n"
"                                                                              \n"
"  /* trilinear filtering */                                                   \n"
"                                                                              \n"
"  for (y=0; y < height; y++)                                                  \n"
"    for (x=0; x < width; x++)                                                 \n"
"      for(c = 0; c < channels; c++)                                           \n"
"        {                                                                     \n"
"          float xf = (float)(x) / s_sigma + padding_xy;                       \n"
"          float yf = (float)(y) / s_sigma + padding_xy;                       \n"
"          float zf = INPUT(x,y,c) / r_sigma + padding_z;                      \n"
"                                                                              \n"
"          int x1 = CLAMP((int)xf, 0, sw-1);                                   \n"
"          int y1 = CLAMP((int)yf, 0, sh-1);                                   \n"
"          int z1 = CLAMP((int)zf, 0, depth-1);                                \n"
"                                                                              \n"
"          int x2 = CLAMP(x1+1, 0, sw-1);                                      \n"
"          int y2 = CLAMP(y1+1, 0, sh-1);                                      \n"
"          int z2 = CLAMP(z1+1, 0, depth-1);                                   \n"
"                                                                              \n"
"          float x_alpha = xf - x1;                                            \n"
"          float y_alpha = yf - y1;                                            \n"
"          float z_alpha = zf - z1;                                            \n"
"                                                                              \n"
"          float interpolated[2];                                              \n"
"                                                                              \n"
"          g_assert(xf >= 0 && xf < sw);                                       \n"
"          g_assert(yf >= 0 && yf < sh);                                       \n"
"          g_assert(zf >= 0 && zf < depth);                                    \n"
"                                                                              \n"
"          for (i=0; i<2; i++)                                                 \n"
"              interpolated[i] =                                               \n"
"              lerp(lerp(lerp(BLURZ(x1, y1, z1, c, i), BLURZ(x2, y1, z1, c, i), x_alpha),\n"
"                        lerp(BLURZ(x1, y2, z1, c, i), BLURZ(x2, y2, z1, c, i), x_alpha), y_alpha),\n"
"                   lerp(lerp(BLURZ(x1, y1, z2, c, i), BLURZ(x2, y1, z2, c, i), x_alpha),\n"
"                        lerp(BLURZ(x1, y2, z2, c, i), BLURZ(x2, y2, z2, c, i), x_alpha), y_alpha), z_alpha);\n"
"                                                                              \n"
"          smoothed[channels*(y*width+x)+c] = interpolated[0] / interpolated[1];\n"
"        }                                                                     \n"
"                                                                              \n"
"  gegl_buffer_set (dst, dst_rect, 0, babl_format (\"RGBA float\"), smoothed,  \n"
"                   GEGL_AUTO_ROWSTRIDE);                                      \n"
"                                                                              \n"
"  g_free (grid);                                                              \n"
"  g_free (blurx);                                                             \n"
"  g_free (blury);                                                             \n"
"  g_free (blurz);                                                             \n"
"  g_free (input);                                                             \n"
"  g_free (smoothed);                                                          \n"
"}                                                                             \n"
"                                                                              \n"
"#include \"opencl/gegl-cl.h\"                                                 \n"
"#include \"buffer/gegl-buffer-cl-iterator.h\"                                 \n"
"#include \"opencl/bilateral-filter-fast.cl.h\"                                \n"
"                                                                              \n"
"static GeglClRunData *cl_data = NULL;                                         \n"
"                                                                              \n"
"static gboolean                                                               \n"
"cl_bilateral (cl_mem                in_tex,                                   \n"
"              cl_mem                out_tex,                                  \n"
"              const GeglRectangle  *roi,                                      \n"
"              const GeglRectangle  *src_rect,                                 \n"
"              gint                  s_sigma,                                  \n"
"              gfloat                r_sigma)                                  \n"
"{                                                                             \n"
"  cl_int cl_err = 0;                                                          \n"
"                                                                              \n"
"  gint c;                                                                     \n"
"                                                                              \n"
"  const gint width    = src_rect->width;                                      \n"
"  const gint height   = src_rect->height;                                     \n"
"                                                                              \n"
"  const gint sw = (width -1) / s_sigma + 1;                                   \n"
"  const gint sh = (height-1) / s_sigma + 1;                                   \n"
"  const gint depth = (int)(1.0f / r_sigma) + 1;                               \n"
"                                                                              \n"
"  size_t global_ws[2];                                                        \n"
"  size_t local_ws[2];                                                         \n"
"                                                                              \n"
"  cl_mem grid = NULL;                                                         \n"
"  cl_mem blur[4] = {NULL, NULL, NULL, NULL};                                  \n"
"                                                                              \n"
"  if (!cl_data)                                                               \n"
"    {                                                                         \n"
"      const char *kernel_name[] = {\"bilateral_downsample\",                  \n"
"                                   \"bilateral_blur\",                        \n"
"                                   \"bilateral_interpolate\",                 \n"
"                                   NULL};                                     \n"
"      cl_data = gegl_cl_compile_and_build (bilateral_filter_fast_cl_source, kernel_name);\n"
"    }                                                                         \n"
"                                                                              \n"
"  if (!cl_data)                                                               \n"
"    return 1;                                                                 \n"
"                                                                              \n"
"                                                                              \n"
"  grid = gegl_clCreateBuffer (gegl_cl_get_context (),                         \n"
"                              CL_MEM_READ_WRITE,                              \n"
"                              sw * sh * depth * sizeof(cl_float8),            \n"
"                              NULL,                                           \n"
"                              &cl_err);                                       \n"
"  CL_CHECK;                                                                   \n"
"                                                                              \n"
"  for(c = 0; c < 4; c++)                                                      \n"
"    {                                                                         \n"
"      blur[c] = gegl_clCreateBuffer (gegl_cl_get_context (),                  \n"
"                                     CL_MEM_READ_WRITE,                       \n"
"                                     sw * sh * depth * sizeof(cl_float2),     \n"
"                                     NULL, &cl_err);                          \n"
"      CL_CHECK;                                                               \n"
"    }                                                                         \n"
"                                                                              \n"
"  local_ws[0] = 8;                                                            \n"
"  local_ws[1] = 8;                                                            \n"
"                                                                              \n"
"  global_ws[0] = ((sw + local_ws[0] - 1)/local_ws[0])*local_ws[0];            \n"
"  global_ws[1] = ((sh + local_ws[1] - 1)/local_ws[1])*local_ws[1];            \n"
"                                                                              \n"
"  gegl_cl_set_kernel_args (cl_data->kernel[0],                                \n"
"                           sizeof(cl_mem),   &in_tex,                         \n"
"                           sizeof(cl_mem),   &grid,                           \n"
"                           sizeof(cl_int),   &width,                          \n"
"                           sizeof(cl_int),   &height,                         \n"
"                           sizeof(cl_int),   &sw,                             \n"
"                           sizeof(cl_int),   &sh,                             \n"
"                           sizeof(cl_int),   &depth,                          \n"
"                           sizeof(cl_int),   &s_sigma,                        \n"
"                           sizeof(cl_float), &r_sigma,                        \n"
"                           NULL);                                             \n"
"  CL_CHECK;                                                                   \n"
"                                                                              \n"
"  cl_err = gegl_clEnqueueNDRangeKernel (gegl_cl_get_command_queue (),         \n"
"                                        cl_data->kernel[0], 2,                \n"
"                                        NULL, global_ws, local_ws,            \n"
"                                        0, NULL, NULL);                       \n"
"  CL_CHECK;                                                                   \n"
"                                                                              \n"
"  local_ws[0] = 16;                                                           \n"
"  local_ws[1] = 16;                                                           \n"
"                                                                              \n"
"  global_ws[0] = ((sw + local_ws[0] - 1)/local_ws[0])*local_ws[0];            \n"
"  global_ws[1] = ((sh + local_ws[1] - 1)/local_ws[1])*local_ws[1];            \n"
"                                                                              \n"
"  gegl_cl_set_kernel_args (cl_data->kernel[1],                                \n"
"                           sizeof(cl_mem), &grid,                             \n"
"                           sizeof(cl_mem), &blur[0],                          \n"
"                           sizeof(cl_mem), &blur[1],                          \n"
"                           sizeof(cl_mem), &blur[2],                          \n"
"                           sizeof(cl_mem), &blur[3],                          \n"
"                           sizeof(cl_int), &sw,                               \n"
"                           sizeof(cl_int), &sh,                               \n"
"                           sizeof(cl_int), &depth,                            \n"
"                           NULL);                                             \n"
"                                                                              \n"
"  cl_err = gegl_clEnqueueNDRangeKernel (gegl_cl_get_command_queue (),         \n"
"                                        cl_data->kernel[1], 2,                \n"
"                                        NULL, global_ws, local_ws,            \n"
"                                        0, NULL, NULL);                       \n"
"  CL_CHECK;                                                                   \n"
"                                                                              \n"
"  global_ws[0] = width;                                                       \n"
"  global_ws[1] = height;                                                      \n"
"                                                                              \n"
"  gegl_cl_set_kernel_args (cl_data->kernel[2],                                \n"
"                           sizeof(cl_mem),   &in_tex,                         \n"
"                           sizeof(cl_mem),   &blur[0],                        \n"
"                           sizeof(cl_mem),   &blur[1],                        \n"
"                           sizeof(cl_mem),   &blur[2],                        \n"
"                           sizeof(cl_mem),   &blur[3],                        \n"
"                           sizeof(cl_mem),   &out_tex,                        \n"
"                           sizeof(cl_int),   &width,                          \n"
"                           sizeof(cl_int),   &sw,                             \n"
"                           sizeof(cl_int),   &sh,                             \n"
"                           sizeof(cl_int),   &depth,                          \n"
"                           sizeof(cl_int),   &s_sigma,                        \n"
"                           sizeof(cl_float), &r_sigma,                        \n"
"                           NULL);                                             \n"
"  CL_CHECK;                                                                   \n"
"                                                                              \n"
"  cl_err = gegl_clEnqueueNDRangeKernel(gegl_cl_get_command_queue (),          \n"
"                                       cl_data->kernel[2], 2,                 \n"
"                                       NULL, global_ws, NULL,                 \n"
"                                       0, NULL, NULL);                        \n"
"  CL_CHECK;                                                                   \n"
"                                                                              \n"
"  cl_err = gegl_clFinish (gegl_cl_get_command_queue ());                      \n"
"  CL_CHECK;                                                                   \n"
"                                                                              \n"
"  cl_err = gegl_clReleaseMemObject (grid);                                    \n"
"  CL_CHECK_ONLY (cl_err);                                                     \n"
"                                                                              \n"
"  for(c = 0; c < 4; c++)                                                      \n"
"    {                                                                         \n"
"      cl_err = gegl_clReleaseMemObject (blur[c]);                             \n"
"      CL_CHECK_ONLY (cl_err);                                                 \n"
"    }                                                                         \n"
"                                                                              \n"
"  return FALSE;                                                               \n"
"                                                                              \n"
"error:                                                                        \n"
"  if (grid)                                                                   \n"
"    gegl_clReleaseMemObject (grid);                                           \n"
"                                                                              \n"
"  for (c = 0; c < 4; c++)                                                     \n"
"    {                                                                         \n"
"      if (blur[c])                                                            \n"
"        gegl_clReleaseMemObject (blur[c]);                                    \n"
"    }                                                                         \n"
"                                                                              \n"
"  return TRUE;                                                                \n"
"}                                                                             \n"
"                                                                              \n"
"static gboolean                                                               \n"
"bilateral_cl_process (GeglOperation       *operation,                         \n"
"                      GeglBuffer          *input,                             \n"
"                      GeglBuffer          *output,                            \n"
"                      const GeglRectangle *result,                            \n"
"                      gint                 s_sigma,                           \n"
"                      gfloat               r_sigma)                           \n"
"{                                                                             \n"
"  const Babl *in_format  = gegl_operation_get_format (operation, \"input\");  \n"
"  const Babl *out_format = gegl_operation_get_format (operation, \"output\"); \n"
"  gint err = 0;                                                               \n"
"                                                                              \n"
"  GeglBufferClIterator *i = gegl_buffer_cl_iterator_new (output,              \n"
"                                                         result,              \n"
"                                                         out_format,          \n"
"                                                         GEGL_CL_BUFFER_WRITE);\n"
"                                                                              \n"
"  gint read = gegl_buffer_cl_iterator_add (i,                                 \n"
"                                           input,                             \n"
"                                           result,                            \n"
"                                           in_format,                         \n"
"                                           GEGL_CL_BUFFER_READ,               \n"
"                                           GEGL_ABYSS_NONE);                  \n"
"                                                                              \n"
"  while (gegl_buffer_cl_iterator_next (i, &err) && !err)                      \n"
"    {                                                                         \n"
"       err = cl_bilateral (i->tex[read],                                      \n"
"                           i->tex[0],                                         \n"
"                           &i->roi[0],                                        \n"
"                           &i->roi[read],                                     \n"
"                           s_sigma,                                           \n"
"                           r_sigma);                                          \n"
"                                                                              \n"
"      if (err)                                                                \n"
"        {                                                                     \n"
"          gegl_buffer_cl_iterator_stop (i);                                   \n"
"          break;                                                              \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  return !err;                                                                \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"gegl_op_class_init (GeglOpClass *klass)                                       \n"
"{                                                                             \n"
"  GeglOperationClass       *operation_class;                                  \n"
"  GeglOperationFilterClass *filter_class;                                     \n"
"                                                                              \n"
"  operation_class = GEGL_OPERATION_CLASS (klass);                             \n"
"  filter_class    = GEGL_OPERATION_FILTER_CLASS (klass);                      \n"
"                                                                              \n"
"  filter_class->process = bilateral_process;                                  \n"
"                                                                              \n"
"  operation_class->prepare                 = bilateral_prepare;               \n"
"  operation_class->get_required_for_output = bilateral_get_required_for_output;\n"
"  operation_class->get_cached_region       = bilateral_get_cached_region;     \n"
"                                                                              \n"
"  operation_class->opencl_support = FALSE;                                    \n"
"                                                                              \n"
"  gegl_operation_class_set_keys (operation_class,                             \n"
"  \"name\"       , \"gegl:bilateral-filter-fast\",                            \n"
"  \"title\"      , \"Bilateral Box Filter\",                                  \n"
"  \"categories\" , \"enhance:noise-reduction\",                               \n"
"  \"description\",                                                            \n"
"           _(\"A fast approximation of bilateral filter, using a box-filter instead of a gaussian blur.\"),\n"
"        NULL);                                                                \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"#endif                                                                        \n"
;
