/* Ergo, version 3.5, a program for linear scaling electronic structure
 * calculations.
 * Copyright (C) 2016 Elias Rudberg, Emanuel H. Rubensson, Pawel Salek,
 * and Anastasia Kruchinina.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * Primary academic reference:
 * Kohn−Sham Density Functional Theory Electronic Structure Calculations 
 * with Linearly Scaling Computational Time and Memory Usage,
 * Elias Rudberg, Emanuel H. Rubensson, and Pawel Salek,
 * J. Chem. Theory Comput. 7, 340 (2011),
 * <http://dx.doi.org/10.1021/ct100611z>
 * 
 * For further information about Ergo, see <http://www.ergoscf.org>.
 */



#ifndef HEADER_PURIFICATION_GENERAL
#define HEADER_PURIFICATION_GENERAL

#include <iostream>
#include <fstream>
#include <sstream>

#include "matrix_typedefs.h"  // definitions of matrix types and interval type 
#include "realtype.h"         // definitions of types
#include "matrix_utilities.h"
#include "integral_matrix_wrappers.h"
#include "output.h"
#include "matrix_proxy.h"
 
#include "puri_info.h" 
#include "constants.h"
#include "utilities.h"
#include "units.h"

#include "files_dense.h"
#include "files_sparse.h"

typedef ergo_real real;

/****************  DEFINE  ******************/

// number of additional iterations
/* After that stopping criterion says to stop, we save matrix X into the file, 
   perform additional iterations and read matrix X back. 
   It is done just for testing purposes in case we wish to run a few SCF iterations
   and use results in the iterations given by the stopping criterion, 
   but at the same time we want to see what happen after stoping criterion says 
   to stop the iterations. */
#define NUM_ADDITIONAL_ITERATIONS 0

// enable more output
//#define DEBUG_PURI_OUTPUT
#define DEBUG_PURI_OUTPUT_NNZ


// enable printf output instead of output to ergoscf.out
// do it if you need run just recursive expansion, not the whole SCF calculations
// #define ENABLE_PRINTF_OUTPUT


/**********************************/


extern real eucl_acc;
extern real mixed_acc;
extern real TOL_OVERLAPPING_BOUNDS;
extern real THRESHOLD_EIG_TOLERANCE;
extern int EIG_EMPTY;
extern int EIG_SQUARE_INT;
extern int EIG_PROJECTION_INT;
extern int EIG_POWER_INT;
extern int EIG_LANCZOS_INT;


/** PurificationGeneral is an abstract class which provides an
 * interface for SP2, SP2ACC and possibly other recursive expansions.
 *
 * \tparam MatrixType Type of a matrix (ex. symmMatrix). */
template<typename MatrixType>
class PurificationGeneral
{
 public:
  typedef ergo_real real;
  typedef intervalType IntervalType;
  typedef mat::normType NormType;
  typedef std::vector<int> VectorTypeInt;
  typedef std::vector<real> VectorTypeReal;
  typedef typename MatrixType::VectorType VectorType;
  typedef MatrixType MatrixTypeWrapper;
  
  PurificationGeneral()
    { 
      initialized = false; 
      computed_spectrum_bounds = false;

      eigVecHOMO = NULL;
      eigVecLUMO = NULL;  
      eigVecHOMO2 = NULL;
      eigVecLUMO2 = NULL;
    }

  /** Set imporatant parameters for the recursive expansion.
   */
  virtual void initialize(MatrixType &F_,                  /**< [in] Effective Hamiltonian matrix. */
			  IntervalType &lumo_bounds_,      /**< [in] Bounds for lumo eigenvalue. */
			  IntervalType &homo_bounds_,      /**< [in] Bounds for homo eigenvalue. */
			  int maxit_,                      /**< [in] Maximum number of recursive expansion iterations. */
			  real error_sub_,                 /**< [in] Allowed error in invariant subspaces. */
			  real error_eig_,                 /**< [in] Error in eigenvalues (used just in old stopping criterion). */
			  int use_new_stopping_criterion_, /**< [in] Set if want to use new stopping criterion. */
			  NormType norm_truncation,        /**< [in] Truncation norm. */
			  NormType norm_stop_crit,         /**< [in] Stopping criterion norm. */
			  int nocc_                        /**< [in] Number of occupied orbitals. */
			  );


  /** Check is function initialize() is already called.
   */
  virtual bool is_initialized() const { return initialized; }
  
  /** Start recursive expansion.
   */
  virtual void PurificationStart();

  virtual ~PurificationGeneral(){};

  /** Clear all matrices in the class.
      Needed to be called if Chunks and Tasks are used, 
      since we need to delete all ChunkIDs 
      before exiting the program. */
  virtual void clear(){ X.clear(); Xsq.clear(); }


  /** Set spectrum bounds.
   *
   *  Used if we know spectrum bounds or want to compute them outside
   *  the class. */
  void set_spectrum_bounds(real eigmin, real eigmax);

  void gen_matlab_file_norm_diff(const char *filename) const;
  void gen_matlab_file_threshold(const char *filename) const;
  void gen_matlab_file_nnz(const char *filename) const;
  void gen_matlab_file_eigs(const char *filename) const;
  void gen_matlab_file_time(const char *filename) const;
  void gen_matlab_file_comm_error(const char *filename) const;
  void gen_matlab_file_cond_num(const char *filename) const;
  void gen_matlab_file_which_eig_computed(const char *filename) const;


  /** Set parameters for computing eigenvectors. */
  void set_eigenvectors_params(string eigenvectors_method_, 
			       string eigenvectors_iterative_method_,
			       int number_of_eigenvalues_,
			       real eigensolver_accuracy_,
			       int eigensolver_maxiter_,
			       int scf_step_,
			       int use_prev_vector_as_initial_guess_,
			       int try_eigv_on_next_iteration_if_fail_,
			       VectorType * eigVecLUMO_, 
			       VectorType * eigVecHOMO_,
			       VectorType * eigVecLUMO2_, 
			       VectorType * eigVecHOMO2_,
			       IntervalType &lumo2_bounds_, 
			       IntervalType &homo2_bounds_
			       );


  PuriInfo info;    /**< Fill in during purification with useful information. */

  MatrixType X; /**< Matrix X. */

  MatrixType Xsq; /**< Matrix X^2. */


 protected:

  MatrixType F;  /**< Matrix F. Needed for computation of eigenvectors.*/

  MatrixType X_homo, X_lumo;  /**< Save matrix Xi in certain iterations for computing eigenvectors (projection method). */

  void save_matrix_now(string str);

  /** Compute spectrum bounds. */
  void get_spectrum_bounds();
  /** Get matrix X0 by mapping spectrum of F into [0,1] in reverse
   *  order.*/
  void compute_X();
  /** Get eigenvalue bounds for X0. */
  void map_bounds_to_0_1();
  /** Run recursive expansion. */
  virtual void purification_process();

  /** Check stopping criterion (obsolete).
   *
   *  Use stopping criterion based on user defined threshold
   *  values. */
  virtual void check_standard_stopping_criterion(const real XmX2_norm, int &stop);

  /** Check stopping criterion.
   *
   *  The new stopping criterion based on the order of convergence is
   *  used, see article arXiv:1507.02087 "Parameterless stopping
   *  criteria..." */
  virtual void check_new_stopping_criterion(const int it, const real XmX2_norm_it, const real XmX2_norm_itm2, const real XmX2_trace, 
				    int &stop, real &estim_order);

  /** Choose stopping criterion and check it.  */
  virtual void stopping_criterion(IterationInfo &iter_info, int &stop, real &estim_order);


  int get_int_eig_iter_method(string eigenvectors_iterative_method);
  int get_int_eig_method(string eigenvectors_method);

  void compute_eigenvectors_without_diagonalization(int it, IterationInfo &iter_info);
  void compute_eigenvectors_without_diagonalization_last_iter_proj();

  void compute_eigenvector(MatrixType const &M, VectorType* eigVecHOMOorLUMO, int it, bool is_homo);


  virtual real total_subspace_error(int it);

  /** Get machine epsilon. */
  static real get_epsilon()
  {return std::numeric_limits<real>::epsilon();}

  /** Get largest number. */
  static real get_max_double()
  {return std::numeric_limits<real>::max();}

  /** Get smallest number. */
  static real get_min_double()
  {return std::numeric_limits<real>::min();}

  /** Get nnz of X in %. */
  double get_nnz_X(size_t &nnzX /**< Number of nz of X*/)
  {nnzX = X.nnz(); return (double) (((double)nnzX)/(X.get_ncols()*X.get_nrows()) * 100);}

  /** Get nnz of X in %. */
  double get_nnz_X()
  {return (double) (((double)X.nnz())/(X.get_ncols()*X.get_nrows()) * 100);}

  /** Get nnz of X^2 in %. */
  double get_nnz_Xsq(size_t &nnzXsq /**< Number of nz of X^2*/)
  {nnzXsq = Xsq.nnz(); return (double) (((double)nnzXsq)/(Xsq.get_ncols()*Xsq.get_nrows()) * 100);}

  /** Get nnz of X^2 in %. */
  double get_nnz_Xsq()
  {return (double) (((double)Xsq.nnz())/(Xsq.get_ncols()*Xsq.get_nrows()) * 100);}

  /** Get homo and lumo bounds from traces and norms of Xi-Xi^2.
   *
   * Used at the end of the recursive expansion.  See article SIAM
   * J. Sci. Comput., 36(2), B147–B170. */
  void estimate_eigenvalues(IntervalType &homo_bounds_F, IntervalType &lumo_bounds_F,
			    VectorTypeReal &XmX2_norm, VectorTypeReal &XmX2_trace);
  
  void determine_iteration_for_eigenvectors();

  void get_iterations_for_lumo_and_homo(int &chosen_iter_lumo,
					int &chosen_iter_homo);

  void get_interval_with_lambda(real &eigVal, VectorType &eigVec, bool &is_homo, bool &is_lumo, const int iter);
  void get_eigenvalue_of_F_from_eigv_of_Xi(real &eigVal, VectorType &eigVec);

  virtual void set_init_params() = 0;
  virtual void set_truncation_parameters() = 0;
  virtual void truncate_matrix(real &thresh, int it) = 0; 
  virtual void estimate_number_of_iterations(int &estim_num_iter) = 0;  
  virtual void purify_X(const int it) = 0;
  virtual void purify_bounds(const int it) = 0;
  virtual void save_other_iter_info(IterationInfo &iter_info, int it) = 0;
  virtual void apply_inverse_poly_vector(const int it, VectorTypeReal &bounds_from_it) = 0;
  virtual void return_constant_C(const int it, real &Cval) = 0;
  /*virtual real apply_inverse_poly(const int it, real x) = 0;*/
  virtual real apply_poly(const int it, real x) = 0;
  virtual void apply_poly_to_eigs(const int it, real &homo, real &lumo) = 0;
  virtual real compute_derivative(const int it, real x) = 0;


  /* PARAMETERS */

  bool initialized;
  
  int use_new_stopping_criterion; /**< True for new stopping criterion */
  int additional_iterations;     /**< Do a few more iterations after convergence */
  
  int maxit;                     /**< Maximum number of iterations */
  int check_stopping_criterion_iter;  /**< Iteration when to start to check stopping criterion. */

  int nocc; /**<  Number of occupied orbitals */
           


  NormType normPuriTrunc;   /**< Norm used for the truncation of matrices. 
			     * Can be mat::frobNorm, mat::mixedNorm, or mat::euclNorm. */


  NormType normPuriStopCrit;  /**< Norm used in the stopping criterion
			       * Can be mat::frobNorm, mat::mixedNorm, or mat::euclNorm. */


  real error_sub; /**< Allowed error in invariant subspaces. */
  real error_eig; /**< Error in eigenvalues (used just in old stopping criterion). */

  real constant_C;  /**< Asymptotic constant C needed for the new stopping criterion. */

  real gammaStopEstim; /**< Used on the stopping criterion for
			  estimation of eigenvalues from
			  purification */
  

  VectorTypeInt VecPoly;  /**< Polynomials computed in the function
			   * estimated_number_of_iterations()
			   * VecPoly[i] = 1 if we use X=X^2 
			   * VecPoly[i] = 0 if we use X=2X-X^2 
			   * (or their accelerated versions) */
  VectorTypeReal VecGap;  /**< Gap computed using inner homo and lumo bounds on each iteration. */


  /*EIGENVECTORS STAFF*/

  int number_of_eigenvalues;

  IntervalType homo_bounds;      /**< Homo bounds for F and (1-homo) bounds for Xi */
  IntervalType lumo_bounds;      /**< Lumo bounds */

  IntervalType homo_bounds_X0; /**< Initial lumo bounds for X */
  IntervalType lumo_bounds_X0; /**< Initial lumo bounds for X */    

  IntervalType homo_bounds_F;  /**< Initial lumo bounds for F */
  IntervalType lumo_bounds_F;  /**< Initial homo bounds for F */

  IntervalType lumo2_bounds;
  IntervalType homo2_bounds;

  IntervalType lumo2_bounds_F;
  IntervalType homo2_bounds_F;

  IntervalType spectrum_bounds;  /**< Outer bounds for the whole spectrum of F/Xi. */
  bool computed_spectrum_bounds;


  /* Eigenvectors */
  int eigenvectors_method;               /**< Chosen method for computing eigenvectors. */
  int eigenvectors_iterative_method;     /**< Chosen eigensolver. */

  // accuracy of the eigenvalue problem solver
  // when residual is less then this value,stop
  real eigensolver_accuracy;
  // maximum number of iterations for eigensolver
  int eigensolver_maxiter;

  string eigenvectors_method_str;
  string eigenvectors_iterative_method_str; 

  int use_prev_vector_as_initial_guess;

  bool compute_eigenvectors_in_this_SCF_cycle;
  bool try_eigv_on_next_iteration_if_fail;

  VectorType * eigVecLUMO; 
  VectorType * eigVecHOMO;
  VectorType * eigVecLUMO2; 
  VectorType * eigVecHOMO2;

  VectorType eigVecLUMORef;
  VectorType eigVecHOMORef;


  real eigValLUMO;
  real eigValHOMO;
  real eigValLUMO2;
  real eigValHOMO2;

  int iter_for_homo;
  int iter_for_lumo;

  VectorTypeInt good_iterations_homo;  /**< Iterations where homo eigenvector can be computed. */
  VectorTypeInt good_iterations_lumo;  /**< Iterations where homo eigenvector can be computed. */

  VectorTypeInt really_good_iterations_homo;   /**< Iterations where homo eigenvector is actually computed. */
  VectorTypeInt really_good_iterations_lumo;   /**< Iterations where lumo eigenvector is actually computed.  */

  int scf_step;
};

/**************************************/


/******************* INIT *******************/


template<typename MatrixType>
void PurificationGeneral<MatrixType>::initialize(MatrixType &F_,
						 IntervalType &lumo_bounds_, 
						 IntervalType &homo_bounds_, 
						 int maxit_, 
						 real error_sub_, 
						 real error_eig_,
						 int use_new_stopping_criterion_,
						 NormType norm_truncation_,
						 NormType norm_stop_crit_,
						 int nocc_
						 )
{
  X = F_;
  lumo_bounds = lumo_bounds_; 
  homo_bounds = homo_bounds_;
  maxit = maxit_;
  assert(maxit >= 1);
  error_sub = error_sub_;
  error_eig = error_eig_;
  use_new_stopping_criterion = use_new_stopping_criterion_;
  normPuriTrunc = norm_truncation_;
  normPuriStopCrit = norm_stop_crit_;
  nocc = nocc_;

  initialized = true;

  // save bounds for the matrix F
  lumo_bounds_F = lumo_bounds_; 
  homo_bounds_F = homo_bounds_;

  /* Use this function to enable printf output of the purification
     work if you want to run just the purification, not the whole scf calculations */
#ifdef ENABLE_PRINTF_OUTPUT
  enable_printf_output();
#endif

  size_t nnzX;
  double nnzXp = get_nnz_X(nnzX);
  do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF,"Creating purification object: N = %d , nocc = %d , NNZ = %lu  <-> %.5lf %%", X.get_nrows(), nocc, nnzX, nnzXp);

#ifdef DEBUG_PURI_OUTPUT
  do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF,"Chosen norm for the truncation: ");
  switch(normPuriTrunc)
    {
    case mat::mixedNorm:  do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF,"mixed"); break;
    case mat::euclNorm:   do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF,"eucl"); break;
    case mat::frobNorm:   do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF,"frob"); break;
    default:
      do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF,"unknown");
    }

  do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF,"Chosen norm for the stopping criterion: ");
  switch(normPuriStopCrit)
    {
    case mat::mixedNorm:  do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF,"mixed"); break;
    case mat::euclNorm:   do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF,"eucl"); break;
    case mat::frobNorm:   do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF,"frob"); break;
    default:
      do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF,"unknown");
    }

  if( this->use_new_stopping_criterion == 1 )
    {
      do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF,"Chosen the NEW stopping criterion.");
      do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF,"Allowed error in subspace %e", (double)error_sub);
    }
  else
    {
      do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Chosen the OLD stopping criterion.");
      do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Allowed error in subspace %e , in eigenvalues %e", (double)error_sub, (double)error_eig);
    }
#endif

  // we need values on iteration it-2 for the stopping criterion
  check_stopping_criterion_iter = 2;

  set_init_params();
  
  additional_iterations = NUM_ADDITIONAL_ITERATIONS;
  info.stopping_criterion = use_new_stopping_criterion; // 1 if new, 0 if not
  info.error_subspace = error_sub;

  number_of_eigenvalues = 0;

  info.debug_output = 0;
#ifdef DEBUG_PURI_OUTPUT 
  info.debug_output = 1;
#endif

}


/*******************************************************/


template<typename MatrixType>
void PurificationGeneral<MatrixType>::set_eigenvectors_params(string eigenvectors_method_, 
							      string eigenvectors_iterative_method_,
							      int number_of_eigenvalues_,
							      real eigensolver_accuracy_,
							      int eigensolver_maxiter_,
							      int scf_step_,
							      int use_prev_vector_as_initial_guess_,
							      int try_eigv_on_next_iteration_if_fail_,
							      VectorType * eigVecLUMO_, 
 							      VectorType * eigVecHOMO_,
							      VectorType * eigVecLUMO2_, 
							      VectorType * eigVecHOMO2_,
							      IntervalType &lumo2_bounds_, 
							      IntervalType &homo2_bounds_
							      )
{

  number_of_eigenvalues = number_of_eigenvalues_;

  if(number_of_eigenvalues > 1)
    throw "Error in set_eigenvectors_params() : cannot compute more than 1 eigenpair.";

  // NULL, empty or containing eigenvector from the previos cycle
  eigVecLUMO = eigVecLUMO_; 
  eigVecHOMO = eigVecHOMO_; 

  eigensolver_accuracy = eigensolver_accuracy_;
  eigensolver_maxiter = eigensolver_maxiter_;
  assert(eigensolver_maxiter >= 1);
  scf_step = scf_step_;
  eigenvectors_method_str = eigenvectors_method_;
  eigenvectors_iterative_method_str = eigenvectors_iterative_method_;
  eigenvectors_method = get_int_eig_method(eigenvectors_method_);
  eigenvectors_iterative_method = get_int_eig_iter_method(eigenvectors_iterative_method_);
  use_prev_vector_as_initial_guess = use_prev_vector_as_initial_guess_;
  try_eigv_on_next_iteration_if_fail = try_eigv_on_next_iteration_if_fail_;
  lumo2_bounds = lumo2_bounds_;
  homo2_bounds = homo2_bounds_;
  lumo2_bounds_F = lumo2_bounds_;
  homo2_bounds_F = homo2_bounds_;

  info.lumo_eigenvector_is_computed = false;
  info.homo_eigenvector_is_computed= false;
  // we hope to be able to compute also HOMO2 and LUMO2
  info.lumo2_eigenvector_is_computed = false;
  info.homo2_eigenvector_is_computed= false;

  iter_for_homo = -1;
  iter_for_lumo = -1;

  // no given method for computing eigenvectors
  if((eigVecLUMO != NULL || eigVecHOMO != NULL) && eigenvectors_method == EIG_EMPTY)
    {
#ifdef DEBUG_PURI_OUTPUT
      do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF,"No given method for computing eigenvectors."
		"Eigenvectors will not be computed in this SCF cycle. Set eigenvectors to NULL.");
#endif
      delete eigVecLUMO; // not NULL here
      delete eigVecHOMO; // not NULL here
      eigVecLUMO = NULL; 
      eigVecHOMO = NULL;
    }
  else
    {
      do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF,"Chosen method to compute eigenvectors: %s", eigenvectors_method_str.c_str());
      do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF,"Chosen iterative method to compute eigenvectors: %s", eigenvectors_iterative_method_str.c_str());
      do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF,"Chosen eigensolver accuracy: %g", (double)eigensolver_accuracy);
    }

  // reuse eigenvector computed in the previous SCF cycle as an initial guess
  if( (eigVecLUMO != NULL || eigVecHOMO != NULL) && use_prev_vector_as_initial_guess)
    do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF,"Use eigenvectors from the previous SCF cycle as an initial guess for this SCF cycle");
  
}

template<typename MatrixType>
int PurificationGeneral<MatrixType>::get_int_eig_method(string eigenvectors_method)
{
  if(eigenvectors_method == "square")     return EIG_SQUARE_INT;
  if(eigenvectors_method == "projection") return EIG_PROJECTION_INT;
  if(eigenvectors_method == "")           return EIG_EMPTY;
  throw "Error in get_int_eig_method(): unknown method to compute eigenvectors";
}


template<typename MatrixType>
int PurificationGeneral<MatrixType>::get_int_eig_iter_method(string eigenvectors_iterative_method)
{
  if(eigenvectors_iterative_method == "power")   return EIG_POWER_INT;
  if(eigenvectors_iterative_method == "lanczos") return EIG_LANCZOS_INT;
  if(eigenvectors_iterative_method == "")        return EIG_EMPTY;
  throw "Error in get_int_eig_iter_method(): unknown iterative method to compute eigenvectors";
}

/****************** PURIFICATION_START ********************/


template<typename MatrixType>
void PurificationGeneral<MatrixType>::PurificationStart()
{

  if(!is_initialized())
    throw "Error in PurificationStart() : function is called for an uninitialized class.";

#ifdef DEBUG_PURI_OUTPUT
  do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Enter PurificationStart()");
#endif

  Util::TimeMeter total_time;// measure total time of the purification process

  compute_eigenvectors_in_this_SCF_cycle = false; // initial value, later in the code may be changed 

  if(!computed_spectrum_bounds)
    {
      Util::TimeMeter total_time_spectrum_bounds;
      get_spectrum_bounds();
      total_time_spectrum_bounds.print(LOG_AREA_DENSFROMF, "get_spectrum_bounds");
      double total_time_spectrum_bounds_stop = total_time_spectrum_bounds.get_elapsed_wall_seconds();
      info.time_spectrum_bounds = total_time_spectrum_bounds_stop;
      computed_spectrum_bounds = true;
    }

  info.set_spectrum_bounds( spectrum_bounds.low(), spectrum_bounds.upp());
  do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Spectrum of F: \t [ %.12lf , %.12lf ]", (double)spectrum_bounds.low(), (double)spectrum_bounds.upp());

  map_bounds_to_0_1();

  if(eigVecLUMO != NULL || eigVecHOMO != NULL) {
    // check if we have non-overlapping homo and lumo bounds
    if(1 - homo_bounds.upp() - lumo_bounds.upp() >= TOL_OVERLAPPING_BOUNDS)
      {
	compute_eigenvectors_in_this_SCF_cycle = true;
      }
    else
      do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Homo and lumo inner bounds are too close (< %g), "
		"homo and lumo eigenvectors will not be computed", (double)TOL_OVERLAPPING_BOUNDS);
  }
    
  info.compute_eigenvectors_in_this_SCF_cycle = compute_eigenvectors_in_this_SCF_cycle;

  if(compute_eigenvectors_in_this_SCF_cycle)
    {
      F = X; // Save original matrix F, needed for computation of the Rayleigh quotient.
      // Quotient is needed to compute eigenvalue corresponding to the eigenvector
      // and see which eigenvalue it is, homo or lumo. 
      F.writeToFile();
    }
           
  
  compute_X(); // F->X, put eigenvalues to the [0,1]

  set_truncation_parameters();


  if(compute_eigenvectors_in_this_SCF_cycle)
    {
      do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF,"calling determine_iteration_for_eigenvectors()");
      determine_iteration_for_eigenvectors();  // on which iterations we should compute eigenvectors
    }

#ifdef DEBUG_PURI_OUTPUT
  do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Starting recursive expansion");
#endif

  purification_process();

  total_time.print(LOG_AREA_DENSFROMF, "Recursive expansion");
  double total_time_stop =  total_time.get_elapsed_wall_seconds();
  info.total_time = total_time_stop;
}


template<typename MatrixType>   
void PurificationGeneral<MatrixType>::purification_process()
{
 
  int it;
  int stop = 0;
  real ONE = 1.0; 
  real thresh;
  real Xsquare_time_stop = -1, total_time_stop = -1, trunc_time_stop = -1, purify_time_stop = -1;
  real frob_diff_time_stop = -1, eucl_diff_time_stop = -1, trace_diff_time_stop = -1, mixed_diff_time_stop = -1, stopping_criterion_time_stop = -1;
  int maxit_tmp = maxit;
  real estim_order = -1;
  real XmX2_trace = -1;
  real XmX2_euc_lower_bound_it = -1;
  real XmX2_fro_norm = -1;
  real XmX2_mixed_norm = -1;
  real XmX2_fro_norm_itm2 = -1;
  real XmX2_norm_itm2 = -1, XmX2_norm_it = -1;
  real K_n = -1, K_nm1 = -1;
	      
  real XmX2_eucl = -1, XmX2_eucl_itm2 = -1;

  int already_stop_before = 0; // flag for checking stopping criterion, needed in case additional_iterations != 0


  IterationInfo iter_info; // 0-th iterations

  // 0 iteration
  it = 0;


  do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "BEFORE ITERATIONS:");	

  Util::TimeMeter total_time; // for this iteration

#ifdef DEBUG_PURI_OUTPUT_NNZ
  double nnzX = get_nnz_X();
#endif

  // truncate matrix
  Util::TimeMeter trunc_time;
  truncate_matrix(thresh, it);
  trunc_time.print(LOG_AREA_DENSFROMF, "truncate_matrix");
  trunc_time_stop = trunc_time.get_elapsed_wall_seconds();

#ifdef DEBUG_PURI_OUTPUT_NNZ
  double nnzXafter = get_nnz_X();
  do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF,"Actual introduced error   %e , nnz before   %.2lf %% , nnz after   %.2lf %%", (double)thresh, nnzX, nnzXafter);
#else
  do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF,"Actual introduced error   %e", (double)thresh);
#endif

  output_current_memory_usage(LOG_AREA_DENSFROMF, "Before X square");

  Util::TimeMeter Xsquare_time;
  X.square(Xsq);
  Xsquare_time.print(LOG_AREA_DENSFROMF, "square");
  Xsquare_time_stop = Xsquare_time.get_elapsed_wall_seconds();

#ifdef DEBUG_PURI_OUTPUT_NNZ
  size_t nnzXsq;
  double nnzXsqp = get_nnz_Xsq(nnzXsq);
  do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "NNZ Xsq = %lu <-> %.2lf %%", nnzXsq, nnzXsqp);    
#endif


  // compute frob norm of X-X2
  Util::TimeMeter frob_diff_time;
  XmX2_fro_norm = MatrixType::frob_diff(X, Xsq);
#ifdef DEBUG_PURI_OUTPUT
  frob_diff_time.print(LOG_AREA_DENSFROMF, "Frobenius norm of X-X^2");
#endif
  frob_diff_time_stop = frob_diff_time.get_elapsed_wall_seconds();
 
  if( normPuriStopCrit == mat::mixedNorm )
    {
      Util::TimeMeter mixed_diff_time;
      XmX2_mixed_norm = MatrixType::mixed_diff(X, Xsq, mixed_acc);
#ifdef DEBUG_PURI_OUTPUT
      mixed_diff_time.print(LOG_AREA_DENSFROMF, "Mixed norm of X-X^2");
#endif
      mixed_diff_time_stop = mixed_diff_time.get_elapsed_wall_seconds();
    }

  // compute trace of X-X2
  Util::TimeMeter trace_diff_time;
  XmX2_trace = MatrixType::trace_diff(X, Xsq);
#ifdef DEBUG_PURI_OUTPUT
  trace_diff_time.print(LOG_AREA_DENSFROMF, "Trace of X-X^2");
#endif
  trace_diff_time_stop = trace_diff_time.get_elapsed_wall_seconds();
  
  if( normPuriStopCrit == mat::euclNorm )
    {
      Util::TimeMeter eucl_diff_time;
      XmX2_eucl = MatrixType::eucl_diff(X, Xsq, eucl_acc);
#ifdef DEBUG_PURI_OUTPUT
      eucl_diff_time.print(LOG_AREA_DENSFROMF, "Spectral norm of X-X^2");
#endif
      eucl_diff_time_stop = eucl_diff_time.get_elapsed_wall_seconds();
    }
  
  
#ifdef DEBUG_PURI_OUTPUT
  if( normPuriStopCrit == mat::euclNorm )
    do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "||X-X^2||_2 = %e", (double)XmX2_eucl);
  
  if( normPuriStopCrit == mat::mixedNorm )
    do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "||X-X^2||_F = %e , ||X-X^2||_mixed = %e", (double)XmX2_fro_norm,  (double)XmX2_mixed_norm);

  if( normPuriStopCrit == mat::frobNorm )
    do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "||X-X^2||_F = %e", (double)XmX2_fro_norm);
  
  do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "trace(X-X^2) = %e", (double)XmX2_trace);
  do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "trace(X) = %e", (double)X.trace());
#endif


  if(compute_eigenvectors_in_this_SCF_cycle)
    compute_eigenvectors_without_diagonalization(it, iter_info);

  ostringstream str_out;
  str_out << "Iteration " << it; 
  total_time.print(LOG_AREA_DENSFROMF, str_out.str().c_str());
  total_time_stop = total_time.get_elapsed_wall_seconds();

  // save some information
  iter_info.it = it;
  iter_info.gap = 1 - homo_bounds.upp() - lumo_bounds.upp(); // = VecGap[0]
  iter_info.threshold_X = thresh;
  iter_info.Xsquare_time = Xsquare_time_stop;
  iter_info.trunc_time = trunc_time_stop;
  iter_info.purify_time = 0;
  iter_info.NNZ_X = get_nnz_X();
  iter_info.NNZ_X2 = get_nnz_Xsq();
  iter_info.XmX2_fro_norm = XmX2_fro_norm;
  iter_info.XmX2_eucl = XmX2_eucl;
  iter_info.XmX2_mixed_norm = XmX2_mixed_norm;
  iter_info.XmX2_trace = XmX2_trace;
  iter_info.total_time = total_time_stop;
  iter_info.homo_bound_low = homo_bounds.low();
  iter_info.lumo_bound_low = lumo_bounds.low();
  iter_info.homo_bound_upp = homo_bounds.upp();
  iter_info.lumo_bound_upp = lumo_bounds.upp();
  stopping_criterion_time_stop = 0; // we are not checking stopping criterion on the 1 iteration
  iter_info.stopping_criterion_time = stopping_criterion_time_stop;
  iter_info.eucl_diff_time = eucl_diff_time_stop;
  iter_info.frob_diff_time = frob_diff_time_stop;
  iter_info.mixed_diff_time = mixed_diff_time_stop;
  iter_info.trace_diff_time = trace_diff_time_stop;

  save_other_iter_info(iter_info, it);
  /**************/

  info.Iterations.push_back(iter_info); // add info about 0 iteration


  output_current_memory_usage(LOG_AREA_DENSFROMF, "Before iteration 1");
  Util::TimeMeter timeMeterWriteAndReadAll;
  std::string sizesStr = mat::FileWritable::writeAndReadAll();
  timeMeterWriteAndReadAll.print(LOG_AREA_DENSFROMF, "FileWritable::writeAndReadAll");
  do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, ((std::string)"writeAndReadAll sizesStr: '" + sizesStr).c_str());


  it = 1;
  while(it <= maxit_tmp)
    {
      do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "ITERATION %d :", it);

      IterationInfo iter_info; // i-th iteration

      Util::TimeMeter total_time; // for this iteration

      Util::TimeMeter purify_time;
      purify_X(it);
      purify_time.print(LOG_AREA_DENSFROMF, "purify_X");
      purify_time_stop = purify_time.get_elapsed_wall_seconds();


#ifdef DEBUG_PURI_OUTPUT_NNZ
       double nnzX = get_nnz_X();
#endif

      // truncate matrix
       Util::TimeMeter trunc_time;
       truncate_matrix(thresh, it);
       trunc_time.print(LOG_AREA_DENSFROMF, "truncate_matrix");
       trunc_time_stop = trunc_time.get_elapsed_wall_seconds();

#ifdef DEBUG_PURI_OUTPUT_NNZ
       double nnzXafter = get_nnz_X();
       do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF,"Actual introduced error   %e , nnz before   %.2lf %% , nnz after   %.2lf %%", thresh, nnzX, nnzXafter);
#else
       do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF,"Actual introduced error   %e", thresh);
#endif


      output_current_memory_usage(LOG_AREA_DENSFROMF, "Before X square");
      
      Util::TimeMeter Xsquare_time;
      X.square(Xsq);
      Xsquare_time.print(LOG_AREA_DENSFROMF, "square");
      Xsquare_time_stop = Xsquare_time.get_elapsed_wall_seconds();
#ifdef DEBUG_PURI_OUTPUT_NNZ
      size_t nnzXsq;
      double nnzXsqp = get_nnz_Xsq(nnzXsq);
      do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "NNZ Xsq = %lu <-> %.2lf %%", nnzXsq, nnzXsqp);  
#endif


      // update bounds for homo and lumo using corresponding polynomial
      purify_bounds(it);

      // compute frob norm of X-X2
      Util::TimeMeter frob_diff_time;
      XmX2_fro_norm = MatrixType::frob_diff(X, Xsq);
#ifdef DEBUG_PURI_OUTPUT
      frob_diff_time.print(LOG_AREA_DENSFROMF, "Frobenius norm of X-X^2");
#endif
      frob_diff_time_stop = frob_diff_time.get_elapsed_wall_seconds();
      
      if( normPuriStopCrit == mat::mixedNorm )
	{
	  Util::TimeMeter mixed_diff_time;
	  XmX2_mixed_norm = MatrixType::mixed_diff(X, Xsq, mixed_acc);
#ifdef DEBUG_PURI_OUTPUT
	  mixed_diff_time.print(LOG_AREA_DENSFROMF, "Mixed norm of X-X^2");
#endif
	  mixed_diff_time_stop = mixed_diff_time.get_elapsed_wall_seconds();
	}
      
      // compute trace of X-X2
      Util::TimeMeter trace_diff_time;
      XmX2_trace = MatrixType::trace_diff(X, Xsq);
#ifdef DEBUG_PURI_OUTPUT
      trace_diff_time.print(LOG_AREA_DENSFROMF, "Trace of X-X^2");
#endif
      trace_diff_time_stop = trace_diff_time.get_elapsed_wall_seconds();


      // CHECK FOR A NEGATIVE TRACE
      if(XmX2_trace < -1e10) // here is definitively some misconvergence
	throw("Error in purification_process() : trace of X-X^2 is negative, seems as a"
	      " misconvergence of the recursive expansion.");
      
      
#ifdef DEBUG_PURI_OUTPUT
      if( normPuriStopCrit == mat::mixedNorm )
	do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "||X-X^2||_F = %e , ||X-X^2||_mixed = %e", XmX2_fro_norm, XmX2_mixed_norm);
      if( normPuriStopCrit == mat::frobNorm )
	do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "||X-X^2||_F = %e", XmX2_fro_norm);

      do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "trace(X-X^2) = %e", XmX2_trace);
      do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "trace(X) = %e", X.trace());
#endif      
      
      if( normPuriStopCrit == mat::euclNorm )
	{
	  Util::TimeMeter eucl_diff_time;
	  XmX2_eucl = MatrixType::eucl_diff(X, Xsq, eucl_acc);
#ifdef DEBUG_PURI_OUTPUT
	  eucl_diff_time.print(LOG_AREA_DENSFROMF, "Spectral norm of X-X^2");
#endif 
	  eucl_diff_time_stop = eucl_diff_time.get_elapsed_wall_seconds();
	  
#ifdef DEBUG_PURI_OUTPUT
	  do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "||X-X^2||_2 = %e", XmX2_eucl);
	  
	  // compute ratio K_{i-2}^2/K_i, see article 
	  // arXiv:1507.02087 "Parameterless stopping criteria..."
	  if(it >=2 )
	    {
	      K_n = XmX2_fro_norm/XmX2_eucl;
	      do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "K_n  = %e ", K_n);
	      K_nm1 = info.Iterations[it-2].XmX2_fro_norm / info.Iterations[it-2].XmX2_eucl;
	      do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "K_{n-1}^2 / K_n = %e ", K_nm1*K_nm1/K_n);
	    }
#endif
	  
	  iter_info.Kratio = K_nm1*K_nm1/K_n;
	}
      

      iter_info.XmX2_fro_norm = XmX2_fro_norm;
      iter_info.XmX2_mixed_norm = XmX2_mixed_norm;
      iter_info.XmX2_trace = XmX2_trace;
      iter_info.XmX2_eucl = XmX2_eucl;
      
      // save some information
      iter_info.it = it;     
      iter_info.threshold_X = thresh;
      iter_info.Xsquare_time = Xsquare_time_stop;
      iter_info.trunc_time = trunc_time_stop;
      iter_info.purify_time = purify_time_stop;
      
      iter_info.eucl_diff_time = eucl_diff_time_stop;
      iter_info.frob_diff_time = frob_diff_time_stop;
      iter_info.mixed_diff_time = mixed_diff_time_stop;
      iter_info.trace_diff_time = trace_diff_time_stop;

      

      if(compute_eigenvectors_in_this_SCF_cycle)
	compute_eigenvectors_without_diagonalization(it, iter_info);

      // check stopping criterion (call function on every iteration
      // larger or equal to check_stopping_criterion_iter)
      if(it >= check_stopping_criterion_iter)
	{
	  Util::TimeMeter stopping_criterion_time;
	  stopping_criterion(iter_info, stop, estim_order);
	  stopping_criterion_time.print(LOG_AREA_DENSFROMF, "stopping_criterion");
	  stopping_criterion_time_stop = stopping_criterion_time.get_elapsed_wall_seconds();
	  iter_info.stopping_criterion_time = stopping_criterion_time_stop;
	}
      else
	{ stop = 0; estim_order = -1; }
      
      // if we reach stopping iteration or maximum allowed number or iterations
      // and we are not already stop (in case we have additional_iterations != 0)
      if( (already_stop_before == 0) && (stop == 1 || it == maxit))
	{ 	   
	  if(stop == 1)
	    {
	      do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "PURIFICATION CONVERGED after %d iterations", it);
	      info.converged = 1;
	    }
	  else // if it == maxit
	    {
	      do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "NOT CONVERGED. Reached the maximum number of iterations %d", maxit);
	      info.converged = 0;
	    }   
	  
	  assert(maxit_tmp <= (int)VecPoly.size());
	  maxit_tmp = it + additional_iterations;
	  already_stop_before = 1;

	  /*
	    if(additional_iterations != 0)
	    {
	    // save matrix
	    do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Save matrix X into the file X_tmp.txt");
	    vector<real> A;
	    symmMatrix Y;
	    X.get_matrix(Y);
	    Y.fullMatrix(A);   
	    write_matrix(saved_matrix_name, A, X.get_nrows(), X.get_ncols());
	    }
	  */	

	}
 
      ostringstream str_out;
      str_out << "Iteration " << it; 
      total_time.print(LOG_AREA_DENSFROMF, str_out.str().c_str());
      total_time_stop = total_time.get_elapsed_wall_seconds();



      /******************/

      
      iter_info.NNZ_X = get_nnz_X();
      iter_info.NNZ_X2 = get_nnz_Xsq();

      iter_info.homo_bound_low = homo_bounds.low();
      iter_info.homo_bound_upp = homo_bounds.upp();
      iter_info.lumo_bound_low = lumo_bounds.low();
      iter_info.lumo_bound_upp = lumo_bounds.upp();

      iter_info.total_time = total_time_stop;
      iter_info.constantC = constant_C;
      if( use_new_stopping_criterion )
	iter_info.order = estim_order;

      save_other_iter_info(iter_info, it);

      /*******************/

      info.Iterations.push_back(iter_info); // add info about i-th iteration to the info 
      
      it++;      
    } //while

  output_current_memory_usage(LOG_AREA_DENSFROMF, "After the last iteration");
  Util::TimeMeter timeMeterWriteAndReadAll_end;
  sizesStr = mat::FileWritable::writeAndReadAll();
  timeMeterWriteAndReadAll_end.print(LOG_AREA_DENSFROMF, "FileWritable::writeAndReadAll");
  do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, ((std::string)"writeAndReadAll sizesStr: '" + sizesStr).c_str());


  info.total_it = maxit_tmp;
  info.additional_iterations = additional_iterations;  

  real acc_err_sub = total_subspace_error(maxit_tmp - additional_iterations);
#ifdef DEBUG_PURI_OUTPUT
  if(acc_err_sub != -1)
    do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "TOTAL accumulated subspace error is %e", acc_err_sub);
#endif
  info.accumulated_error_subspace = acc_err_sub;

  IntervalType homo_estim_bounds_F, lumo_estim_bounds_F;
  homo_estim_bounds_F = intervalType(-1e22,1e22);
  lumo_estim_bounds_F = intervalType(-1e22,1e22);

  IntervalType homo2_estim_bounds_F, lumo2_estim_bounds_F;
  homo2_estim_bounds_F = intervalType(-1e22,1e22);
  lumo2_estim_bounds_F = intervalType(-1e22,1e22);



  if(info.converged == 1)
    {
      // estimate eigenvalues of the matrix F
      VectorTypeReal norms, traces;
      info.get_vec_frob_norms(norms);      
      info.get_vec_traces(traces);
      estimate_eigenvalues(homo_estim_bounds_F, lumo_estim_bounds_F, norms, traces);


      do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Estimated bounds for the eigenvalues for the Fock matrix:");
      do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "ESTIM LUMO: [ %.12lf , %.12lf ]", lumo_estim_bounds_F.low(), lumo_estim_bounds_F.upp());
      do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "ESTIM HOMO: [ %.12lf , %.12lf ]", homo_estim_bounds_F.low(), homo_estim_bounds_F.upp());
    }
  else
    do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Cannot estimate eigenvalues since the purification did not converge");


  info.homo_estim_low_F = homo_estim_bounds_F.low();
  info.homo_estim_upp_F = homo_estim_bounds_F.upp();
  info.lumo_estim_low_F = lumo_estim_bounds_F.low();
  info.lumo_estim_upp_F = lumo_estim_bounds_F.upp();

  info.homo2_estim_low_F = homo2_estim_bounds_F.low();
  info.homo2_estim_upp_F = homo2_estim_bounds_F.upp();
  info.lumo2_estim_low_F = lumo2_estim_bounds_F.low();
  info.lumo2_estim_upp_F = lumo2_estim_bounds_F.upp();


 
  if(compute_eigenvectors_in_this_SCF_cycle && eigenvectors_method == EIG_PROJECTION_INT)
    compute_eigenvectors_without_diagonalization_last_iter_proj();
 
  // save HOMO and LUMO eigenvalues in the info object
  if(compute_eigenvectors_in_this_SCF_cycle)
    {
      int ITER_DIFF = 2;
      // check if we passed iter_for_homo iteration
      if(!eigVecHOMO->is_empty()  && info.total_it < iter_for_homo)
	{
	  do_output(LOG_CAT_WARNING, LOG_AREA_DENSFROMF, "HOMO eigenvector was not computed. Iteration for homo: %d, total number of iterations: %d",
		    iter_for_homo, info.total_it);
	  info.eigValHOMO = 0;
	  info.homo_eigenvector_is_computed = false;
	  eigVecHOMO->clear_structure();
	}
      else
	{
	  // if yes, was it one of the last iterations?
	  if(!eigVecHOMO->is_empty()  && info.total_it - iter_for_homo < ITER_DIFF && info.homo_eigenvector_is_computed)
	    do_output(LOG_CAT_WARNING, LOG_AREA_DENSFROMF, "HOMO eigenvector was computed in one of the last recursive expansion iterations (%d of total %d). "
		      "Eigenvalues of the matrix X in this iteration probably already reached the level of numerical errors, "
		      "thus result may not be accurate!", iter_for_homo, info.total_it);
	}

      // check if we passed iter_for_lumo iteration
      if(!eigVecLUMO->is_empty()  && info.total_it < iter_for_lumo)
	{
	  do_output(LOG_CAT_WARNING, LOG_AREA_DENSFROMF, "LUMO eigenvector was not computed. Iteration for lumo: %d, total number of iterations: %d",
		    iter_for_lumo, info.total_it);
	  info.eigValLUMO = 0;
	  info.lumo_eigenvector_is_computed = false;
	  eigVecLUMO->clear_structure();
	}
      else
	{
	  // if yes, was it one of the last iterations?
	  if(!eigVecLUMO->is_empty()  && info.total_it - iter_for_lumo < ITER_DIFF && info.lumo_eigenvector_is_computed)
	    do_output(LOG_CAT_WARNING, LOG_AREA_DENSFROMF, "LUMO eigenvector was computed in one of the last recursive expansion iterations (%d of total %d). "
		      "Eigenvalues of the matrix X in this iteration probably already reached the level of numerical errors, "
		      "thus result may not be accurate!", iter_for_lumo, info.total_it);
	}
     
      real low_lumo_F_bound = info.lumo_estim_low_F;
      real upp_lumo_F_bound = info.lumo_estim_upp_F;
      real low_homo_F_bound = info.homo_estim_low_F;
      real upp_homo_F_bound = info.homo_estim_upp_F;
      real low_lumo2_F_bound = info.lumo2_estim_low_F;
      real upp_lumo2_F_bound = info.lumo2_estim_upp_F;
      real low_homo2_F_bound = info.homo2_estim_low_F;
      real upp_homo2_F_bound = info.homo2_estim_upp_F;

      // For small cases bounds can be too good or even slightly wrong
      // due to numerical errors; thus we allow a small flexibility
      real flex_tolerance = THRESHOLD_EIG_TOLERANCE;

      if(info.homo_eigenvector_is_computed) // check if eigenvector was computed
	if(low_homo_F_bound - flex_tolerance <= eigValHOMO && eigValHOMO <= upp_homo_F_bound + flex_tolerance)
	  {
	    do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "HOMO eigenvalue is %lf , HOMO bounds are [ %lf , %lf ]", 
		      eigValHOMO, low_homo_F_bound, upp_homo_F_bound);
	    info.eigValHOMO = eigValHOMO;
	  }
	else
	  {
	    do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Computed HOMO eigenvalue is outside HOMO bounds [ %lf , %lf ],"
		      " discard computed HOMO eigenvector.", 
		      low_homo_F_bound, upp_homo_F_bound);
	    info.eigValHOMO = 0;
	    info.homo_eigenvector_is_computed = false;
	    /* NOTE: clear() gives a zero vector, which may not be empty (i.e. still save some data about the structure)!
	     Thus we use clear_structure() to get an empty vector. */
	    eigVecHOMO->clear_structure();
	  }
      else
	{
	  info.eigValHOMO = 0;
	  info.homo_eigenvector_is_computed = false;
	  eigVecHOMO->clear_structure();
	  do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "HOMO eigenvector is not computed.");
	}

      if(info.lumo_eigenvector_is_computed) // check if eigenvector was computed
	if(low_lumo_F_bound - flex_tolerance <= eigValLUMO && eigValLUMO <= upp_lumo_F_bound + flex_tolerance)
	  {
	    do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "LUMO eigenvalue is %lf , LUMO bounds are [ %lf , %lf ]", 
		      eigValLUMO, low_lumo_F_bound, upp_lumo_F_bound);
	    info.eigValLUMO = eigValLUMO;
	  }
	else	
	  {
	    do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Computed LUMO eigenvalue is outside LUMO bounds [ %lf , %lf ],"
		      " discard computed LUMO eigenvector.", 
		      low_lumo_F_bound, upp_lumo_F_bound);
	    info.eigValLUMO = 0;
	    info.lumo_eigenvector_is_computed = false;
	    eigVecLUMO->clear_structure();
	  }
      else
	{
	  info.eigValLUMO = 0;
	  info.lumo_eigenvector_is_computed = false;
	  eigVecLUMO->clear_structure();
	  do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "LUMO eigenvector is not computed.");
	}

      if(info.homo_eigenvector_is_computed && info.lumo_eigenvector_is_computed) {
	real gap = eigValLUMO - eigValHOMO;
	do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Computed HOMO-LUMO gap is %lf = %lf eV", gap, gap / UNIT_one_eV);
      }

    } 


  // end of purification

}



/***************** COMPUTE_X *****************/

template<typename MatrixType>
void PurificationGeneral<MatrixType>::compute_X()
{
  if(spectrum_bounds.empty())
    do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Interval spectrum_bounds is empty in compute_X().");

#ifdef DEBUG_PURI_OUTPUT
  do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Transform matrix F...");  
#endif

  real eigmin = spectrum_bounds.low();
  real eigmax = spectrum_bounds.upp();
  X.add_identity(-eigmax);     
  X.mult_scalar((real)1.0 / (eigmin - eigmax));
}




template<typename MatrixType>
void PurificationGeneral<MatrixType>::map_bounds_to_0_1()
{
  if(spectrum_bounds.empty())
    do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Interval spectrum_bounds is empty in map_bounds_to_0_1().");

#ifdef DEBUG_PURI_OUTPUT
  do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Transform homo and lumo bounds...");  
#endif

  real eigmin = spectrum_bounds.low();
  real eigmax = spectrum_bounds.upp();

  // Compute transformed homo and lumo eigenvalues.   
  // homo and lumo must be in the [lmin, lmax] interval
  homo_bounds.intersect(spectrum_bounds);
  lumo_bounds.intersect(spectrum_bounds);

  homo2_bounds.intersect(spectrum_bounds);
  lumo2_bounds.intersect(spectrum_bounds);

#ifdef DEBUG_PURI_OUTPUT
  if ( homo_bounds.empty() )
    do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Interval homo_bounds is empty.");
  if ( lumo_bounds.empty() )
    do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Interval lumo_bounds is empty.");

  if(!mat::Interval<real>::intersect(homo_bounds,lumo_bounds).empty())
    {
      do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Bounds for homo and lumo of F are not known.");
      //return;
    }
#endif

#ifdef DEBUG_PURI_OUTPUT
  do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Put eigenvalues of F to the interval [0,1] in reverse order.");
#endif
 

  // (1-homo) and lumo bounds for matrix X  
  homo_bounds = (homo_bounds - eigmax) / (eigmin - eigmax);
  homo_bounds = IntervalType(1-homo_bounds.upp(), 1-homo_bounds.low());
  lumo_bounds = (lumo_bounds - eigmax) / (eigmin - eigmax);

  homo_bounds_X0 = homo_bounds;
  lumo_bounds_X0 = lumo_bounds;

  homo2_bounds = (homo2_bounds - eigmax) / (eigmin - eigmax);
  homo2_bounds = IntervalType(1-homo2_bounds.upp(), 1-homo2_bounds.low());
  lumo2_bounds = (lumo2_bounds - eigmax) / (eigmin - eigmax);


#ifdef DEBUG_PURI_OUTPUT
  do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "HOMO bounds of X: \t [ %.12lf , %.12lf ]", 1-homo_bounds.upp(), 1-homo_bounds.low());
  do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "LUMO bounds of X: \t [ %.12lf , %.12lf ]", lumo_bounds.low(), lumo_bounds.upp());
#endif
}


/****************** SET_SPECTRUM_BOUNDS *****************/

template<typename MatrixType>
void PurificationGeneral<MatrixType>::set_spectrum_bounds(real eigmin, real eigmax)
{
  spectrum_bounds = IntervalType(eigmin, eigmax);  
  computed_spectrum_bounds = true;
}


/****************** GET_SPECTRUM_BOUNDS *****************/

template<typename MatrixType> 
void PurificationGeneral<MatrixType>::get_spectrum_bounds()
{
  // find approximations using Gershgorin bounds
  real eigminG, eigmaxG, eigmin, eigmax;

  Util::TimeMeter total_time_spectrum_bounds;
  X.gershgorin(eigminG, eigmaxG);  
  total_time_spectrum_bounds.print(LOG_AREA_DENSFROMF, "gershgorin");

#ifdef DEBUG_PURI_OUTPUT
  do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Gershgorin bounds: [ %.12lf , %.12lf ]", eigminG, eigmaxG);
#endif

  /* ELIAS NOTE 2016-02-08: Expand Gershgorin bounds by a very small
     amount to avoid problems of misconvergence in case one of the
     bounds is exact and the gap is zero (in such cases we want
     convergence failure, not convergence to a solution with wrong
     occupation). */
  real smallNumberToExpandWith = std::sqrt(std::numeric_limits<real>::epsilon());
  eigminG -= smallNumberToExpandWith;
  eigmaxG += smallNumberToExpandWith;
#ifdef DEBUG_PURI_OUTPUT
  do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "EXPANDED Gershgorin bounds: [ %.12lf , %.12lf ]", eigminG, eigmaxG);
#endif

  eigmin = eigminG;
  eigmax = eigmaxG;

  // Lanczos helps us to improve Gershgorin bounds
#if 1 // 0 - without Lanczos, 1 - with Lanczos
#ifndef USE_CHUNKS_AND_TASKS
#ifdef DEBUG_PURI_OUTPUT
  // try to impove with Lanczos algorithm
  
  do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Trying to impove bounds using Lanczos algorithm...");
#endif

  real acc = 1e3*std::sqrt(get_epsilon());// std::sqrt(std::sqrt(get_epsilon())); - this accuracy may be to high for single precision
  MatrixType Xshifted(X);
  Xshifted.add_identity((real)(-1.0) * eigminG); // Xsh = X - eigmin*I

  int maxIter = 100;
  try
    {
      eigmax = Xshifted.eucl(acc, maxIter) + eigminG + acc;
    }
  catch (mat::AcceptableMaxIter & e)
    {
#ifdef DEBUG_PURI_OUTPUT
      do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Lanczos failed to find extreme upper eigenvalue within maxiter... using Gershgorin bound");
#endif
      eigmax = eigmaxG;
    }

  // Now we want to create Fshifted = ( F - lambdaMaxGers*id ) but we
  // do this starting from the existing Fshifted, correcting it back
  // to F and then subtracting lambdaMaxGers*id.
  Xshifted.add_identity((real)( 1.0) * eigminG); // Now Fshifted = F.
  Xshifted.add_identity((real)(-1.0) * eigmaxG);
  
  try
    {
      eigmin = -Xshifted.eucl(acc, maxIter) + eigmaxG - acc;
    }
  catch (mat::AcceptableMaxIter & e)
    {
#ifdef DEBUG_PURI_OUTPUT
      do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Lanczos failed to find extreme lower eigenvalue within maxiter... using Gershgorin bound");
#endif
      eigmin = eigminG;
    }

#endif // USE_CHUNKS_AND_TASKS
#endif

  // extreme case of matrix with 1 element
  if(eigmin == eigmax)
    {
      real tol = 1e-2;
      eigmin -= tol;
      eigmax += tol;
    }

  spectrum_bounds = IntervalType(eigmin, eigmax);  
}


/****************** CHECK_STOPPING_CRITERION  **********************/

template<typename MatrixType>
void PurificationGeneral<MatrixType>::stopping_criterion(IterationInfo &iter_info, int &stop, real &estim_order)
{
  int it = iter_info.it;
  real XmX2_norm_it = -1, XmX2_norm_itm2 = -1;
 
  if(it < check_stopping_criterion_iter) return; // do not check the stopping criterion

  if( use_new_stopping_criterion) 
    {
      // if spectral norm is used for the etimation of the order
      if( normPuriStopCrit == mat::euclNorm )
	{
	  XmX2_norm_it = iter_info.XmX2_eucl;
	  XmX2_norm_itm2 = info.Iterations[it-2].XmX2_eucl;
	}
      else
	if( normPuriStopCrit == mat::frobNorm )
	  {
	    XmX2_norm_it = iter_info.XmX2_fro_norm;
	    XmX2_norm_itm2 = info.Iterations[it-2].XmX2_fro_norm;
	  }
	else
	  if( normPuriStopCrit == mat::mixedNorm )
	    {
	      XmX2_norm_it = iter_info.XmX2_mixed_norm;
	      XmX2_norm_itm2 = info.Iterations[it-2].XmX2_mixed_norm;
	    }
	  else throw "Error in stopping_criterion() : unknown matrix norm.";

      real XmX2_trace = iter_info.XmX2_trace;
      check_new_stopping_criterion(it, XmX2_norm_it, XmX2_norm_itm2, XmX2_trace, stop, estim_order);
    }
  else // use standard stopping criterion
    {
      if( normPuriStopCrit == mat::euclNorm )
	XmX2_norm_it = iter_info.XmX2_eucl;
      if( normPuriStopCrit == mat::frobNorm )
	XmX2_norm_it = iter_info.XmX2_fro_norm;
      if( normPuriStopCrit == mat::mixedNorm )
	XmX2_norm_it = iter_info.XmX2_mixed_norm;

      estim_order = -1;
      check_standard_stopping_criterion(XmX2_norm_it, stop);
    }	  
}



template<typename MatrixType>
void PurificationGeneral<MatrixType>::check_standard_stopping_criterion(const real XmX2_norm, int &stop)
{
#ifdef DEBUG_PURI_OUTPUT
  do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Checking standard stopping criterion...    ");
#endif
  bool homoLumo_converged = (homo_bounds.upp() < error_eig && 
			     lumo_bounds.upp() < error_eig);
  bool XmX2norm_converged = XmX2_norm < error_eig;
  if( homoLumo_converged || XmX2norm_converged) stop = 1;

#ifdef DEBUG_PURI_OUTPUT
  do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "stop = %d", stop);
#endif
}


template<typename MatrixType>
void PurificationGeneral<MatrixType>::check_new_stopping_criterion(const int it, const real XmX2_norm_it, const real XmX2_norm_itm2, const real XmX2_trace, 
								   int &stop, real &estim_order)
{
  // must do at least 2 iterations
  if( it < 2 ) { estim_order = -1; return; }

#ifdef DEBUG_PURI_OUTPUT
  do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Checking stopping criterion...   ");
#endif

  real C; // constant may depend on the purification method
  return_constant_C(it, C); 
  this->constant_C = C;
  if( C == -1 ) { estim_order = -1; return; }

  real epsilon = PurificationGeneral<MatrixType>::get_epsilon();
  // we cannot divide by zero! but the norm is zero just if we converge  
  /*
    NOTE: one can argue that trace of X-X^2 can be negative also before the stagnation phase.
    However, of we assume that all eigenvalues are in [0, 1], then square make them just smaller, 
    thus trace(X-X^2) = sum(lambda_i) - sum(lambda_i^2) >=0.
    In practice, we can have negative eigenvalues of eigenvalues more then 1. But untill we reached 
    the stagnation phase there will be eigenvalues inside [0,1] which are influence the sum(lambda_i) much more. 
    Just when we reched the stagnation phase, eigenvalues outside [0,1] will influence significantly the result.
  */
  if( XmX2_norm_it < epsilon || (XmX2_trace < 0 && fabs(XmX2_trace) < sqrt(epsilon))) 
    {
#ifdef DEBUG_PURI_OUTPUT
      do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "The upper bound for the chosen norm is less then the machine epsilon or trace is negative... stop");
#endif
      estim_order = -1; 
      stop = 1; 
      return; 
    }

  estim_order = log(XmX2_norm_it/C) / log(XmX2_norm_itm2); // rate of convergence
    
  if(VecPoly[it-1] != VecPoly[it]          // alternating polynomials
     && XmX2_norm_itm2 < 1                 // assumption for frob and mixed norms, always true for the spectral norm
     && estim_order <= ORDER )             // r <= 2 (or smaller value)
    {stop = 1;}

#ifdef DEBUG_PURI_OUTPUT
  if(VecPoly[it-1] != VecPoly[it] && XmX2_norm_itm2 < 1)
    do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Order of convergence = %lf, stop = %d", estim_order, stop);
  else
    do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Order of convergence cannot be computed");
#endif
}


/********************* TOTAL_SUBSPACE_ERROR ****************/

template<typename MatrixType>
typename PurificationGeneral<MatrixType>::real 
PurificationGeneral<MatrixType>::total_subspace_error(int it) 
{
  assert(it <= (int)VecGap.size());

  real error = 0;
  real normE;
  for( int i = 0; i <= it; ++i )
    {
      if(VecGap[i] == -1) return -1; // gap is not known

      normE = info.Iterations[i].threshold_X;
      error += normE/(VecGap[i]-normE); 
    }

  return error;
}




template<typename MatrixType>
void PurificationGeneral<MatrixType>::estimate_eigenvalues(IntervalType &homo_bounds_F, IntervalType &lumo_bounds_F,
							   VectorTypeReal &XmX2_norm, VectorTypeReal &XmX2_trace)
{
#ifdef DEBUG_PURI_OUTPUT
  do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Start estimation of eigenvalues of the Fock matrix F.");
#endif

  // do not use additional iterations in the estimation
  int total_it = info.total_it - info.additional_iterations;

  // lumo_out, lumo_in, 1-homo_out, 1-homo_in
  VectorTypeReal bounds_from_it(4);
  VectorTypeReal final_bounds(4, 1); // set all to one

  homo_bounds_F = IntervalType(-1e22, 1e22);
  lumo_bounds_F = IntervalType(-1e22, 1e22);

  // criterion for the eligible iterations for the estimation of the bounds
  real STOP_NORM = gammaStopEstim - gammaStopEstim * gammaStopEstim;
  real vi, wi;
  real LOWER_BOUND;
  real temp_value;


  for(int it = total_it; it >= 0; it--) 
    {
      vi = XmX2_norm[it];
      wi = XmX2_trace[it];
 


      if(vi >= STOP_NORM) 
	{
#ifdef DEBUG_PURI_OUTPUT
	  do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Stop estimation of eigenvalues of the Fock matrix F on iteration %d, Frobenius norm is %e", it, vi);
#endif
	  break;
	}

      if(wi <= sqrt(get_epsilon()))
	{
#ifdef DEBUG_PURI_OUTPUT 
	  do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Trace is too small or negative on iteration %d ! Skip this iteration.", it);
#endif
	  continue;
	}

      // lumo bounds
      bounds_from_it[1] = 2*vi/(1 + std::sqrt(1 - 4*vi));

      temp_value = vi*vi/wi;
      bounds_from_it[0] = 2*temp_value/(1 + std::sqrt(1 - 4*temp_value));

      // bounds for 1-homo
      bounds_from_it[2] = bounds_from_it[0]; 
      bounds_from_it[3] = bounds_from_it[1];
      
      
      apply_inverse_poly_vector(it, bounds_from_it);

      final_bounds[0] = std::min(final_bounds[0], bounds_from_it[0]); // outer
      final_bounds[1] = std::min(final_bounds[1], bounds_from_it[1]); // inner
      
      final_bounds[2] = std::min(final_bounds[2], bounds_from_it[2]); // outer
      final_bounds[3] = std::min(final_bounds[3], bounds_from_it[3]); // inner

    }

  // get bounds for F
  real maxeig = spectrum_bounds.upp();
  real mineig = spectrum_bounds.low();
  
  lumo_bounds_F = IntervalType(maxeig*(1-final_bounds[1]) + mineig*final_bounds[1], 
			       maxeig*(1-final_bounds[0]) + mineig*final_bounds[0]);
  homo_bounds_F = IntervalType(mineig*(1-final_bounds[2]) + maxeig*final_bounds[2], 
			       mineig*(1-final_bounds[3]) + maxeig*final_bounds[3]);
}		     

/*************************** SAVE MATRIX **************************/

template<typename MatrixType>
void PurificationGeneral<MatrixType>::save_matrix_now(string str)
{
  symmMatrix Y;
  X.get_matrix(Y);
  vector<int> Itmp, I, Jtmp, J;
  vector<real> Vtmp, V;
  Y.get_all_values(Itmp, Jtmp, Vtmp);

  size_t nnz = 0;
  // Count nonzeros
  for(size_t i = 0; i < Itmp.size(); i++) 
    nnz += ( Vtmp[i] != 0 );
    
  I.reserve(nnz);
  J.reserve(nnz);
  V.reserve(nnz);
  // Extract nonzeros
  for(size_t i = 0; i < Itmp.size(); i++) 
    {
      if ( Vtmp[i] != 0 ) 
	{
	  I.push_back( Itmp[i] );
	  J.push_back( Jtmp[i] );
	  V.push_back( Vtmp[i] );	
	}
    }
    
  string name  = "X_" + str + ".mtx";   
  if( write_matrix_to_mtx(name.c_str(), I, J, V, X.get_nrows(), X.get_ncols()) == -1)
    throw "Error in save_matrix_now : error in write_matrix_to_mtx.\n";
}



/********************************************************
 ***          COMPUTATION OF EIGENVECTORS       ***
 *********************************************************/

template<typename MatrixType>
void PurificationGeneral<MatrixType>::compute_eigenvectors_without_diagonalization(int it, IterationInfo &iter_info)
{
  real homo_total_time_stop,  lumo_total_time_stop, homo_solver_time_stop, lumo_solver_time_stop;

  /* flags deciding on which iteration to compute eigenvectors */
  bool compute_homo_now = false;
  bool compute_lumo_now = false;
  
  if(eigVecHOMO != NULL)
    if(it == iter_for_homo)  compute_homo_now = true;
  if(eigVecLUMO != NULL)
    if(it == iter_for_lumo)  compute_lumo_now = true;

  if(compute_homo_now && !info.homo_eigenvector_is_computed)
    {
    if(eigenvectors_method == EIG_SQUARE_INT)
      {
	Util::TimeMeter homo_total_time;

	MatrixType M(Xsq); // M = Xsq
	Xsq.writeToFile();

	/* CHOOSING SHIFT */

	real sigma;
	real lumo_tmp;
	if(!info.lumo_eigenvector_is_computed)
	  lumo_tmp = lumo_bounds.upp(); // use inner bound for lumo in this iteration
	else
	  {
	    lumo_tmp = (spectrum_bounds.upp() - eigValLUMO)/(spectrum_bounds.upp() - spectrum_bounds.low());
	    // find lumo eigenvalue in this iteration
	    for(int i = 1; i <= it; ++i)
	      lumo_tmp = apply_poly(i, lumo_tmp);
	  }

	// sigma = (homo_out + lumo_in)/2
	real outer_bound = 1-homo_bounds.low();

	sigma = (outer_bound+lumo_tmp)/2; 
	do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "choose shift %lf", sigma);

	/* GET MATRIX */
	
	M.add_identity(sigma*sigma); // X^2 + sigma^2I
#ifndef USE_CHUNKS_AND_TASKS 
	M.add(X, (real)-2*sigma); // X^2 + sigma^2I - 2sigmaX
#else
	throw "Error in compute_eigenvectors_without_diagonalization() : add(X, alpha) is not implemented with Chunks and Tasks";
#endif
	M.mult_scalar((real)-1.0); // -(X-sigma*I)^2

	Util::TimeMeter homo_solver_time;
	compute_eigenvector(M, eigVecHOMO, it, true);
	homo_solver_time.print(LOG_AREA_DENSFROMF, "compute_eigenvector()");
	homo_solver_time_stop = homo_solver_time.get_elapsed_wall_seconds();

	homo_total_time.print(LOG_AREA_DENSFROMF, "compute homo eigenvector");
	homo_total_time_stop = homo_total_time.get_elapsed_wall_seconds();

	iter_info.homo_eig_solver_time = homo_solver_time_stop;
	iter_info.orbital_homo_time = homo_total_time_stop;
	iter_info.shift_square_method_homo = sigma;


	M.clear();
	Xsq.readFromFile();

      }
    else // Projection method
      {
	Util::TimeMeter homo_time_save_matrix;
	X.writeToFile();
	X_homo = X;
	X.readFromFile();
	homo_time_save_matrix.print(LOG_AREA_DENSFROMF, "saving homo matrix using writeToFile");
      }
    }

  if(compute_lumo_now && !info.lumo_eigenvector_is_computed)
    {
    if(eigenvectors_method == EIG_SQUARE_INT)
      {
	Util::TimeMeter lumo_total_time;

	MatrixType M(Xsq); // M = Xsq
	Xsq.writeToFile();

	/* CHOOSING SHIFT */

	real sigma;
	real homo_tmp;
	if(!info.homo_eigenvector_is_computed)
	  homo_tmp = 1-homo_bounds.upp();
	else
	  {
	    homo_tmp = (spectrum_bounds.upp()-eigValHOMO)/(spectrum_bounds.upp() - spectrum_bounds.low());
	    // find homo eigenvalue in this iteration
	    for(int i = 1; i <= it; ++i)
	      homo_tmp = apply_poly(i, homo_tmp);
	  }

	// sigma = (lumo_out + homo_in)/2
	real outer_bound = lumo_bounds.low();

	sigma = (outer_bound+homo_tmp)/2; 
	do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "choose shift %lf", sigma);

	/* GET MATRIX */

	M.add_identity(sigma*sigma); // X^2 + sigma^2I
#ifndef USE_CHUNKS_AND_TASKS 
	M.add(X, (real)-2*sigma); // X^2 + sigma^2I - 2sigmaX
#else
	throw "Error in compute_eigenvectors_without_diagonalization() : add(X, alpha) is not implemented with Chunks and Tasks";
#endif
	M.mult_scalar((real)-1.0); // -(X-sigma*I)^2

	Util::TimeMeter lumo_solver_time;
	compute_eigenvector(M, eigVecLUMO, it, false);
	lumo_solver_time.print(LOG_AREA_DENSFROMF, "compute_eigenvector()");
	lumo_solver_time_stop = lumo_solver_time.get_elapsed_wall_seconds();

	lumo_total_time.print(LOG_AREA_DENSFROMF, "compute lumo eigenvector");
	lumo_total_time_stop = lumo_total_time.get_elapsed_wall_seconds();

	iter_info.lumo_eig_solver_time = lumo_solver_time_stop;
	iter_info.orbital_lumo_time = lumo_total_time_stop;
	iter_info.shift_square_method_lumo = sigma;

	M.clear();
	Xsq.readFromFile();
      }
    else
      {
	Util::TimeMeter lumo_time_save_matrix;
	X.writeToFile();
	X_lumo = X;
	X.readFromFile();
	lumo_time_save_matrix.print(LOG_AREA_DENSFROMF, "saving lumo matrix using writeToFile");
      }
    }
}


#ifdef USE_CHUNKS_AND_TASKS
template<typename MatrixType>
void PurificationGeneral<MatrixType>::compute_eigenvectors_without_diagonalization_last_iter_proj()
{throw "compute_eigenvectors_without_diagonalization_last_iter_proj() is not implemented for CHTMatrix.";}
#else
template<typename MatrixType>
void PurificationGeneral<MatrixType>::compute_eigenvectors_without_diagonalization_last_iter_proj()
{
  real homo_total_time_stop,  lumo_total_time_stop, homo_solver_time_stop, lumo_solver_time_stop;
  real DX_mult_time_homo_stop, DX_mult_time_lumo_stop;

  output_current_memory_usage(LOG_AREA_DENSFROMF, "Before computing eigenvectors:");
  Util::TimeMeter timeMeterWriteAndReadAll;
  std::string sizesStr = mat::FileWritable::writeAndReadAll();
  timeMeterWriteAndReadAll.print(LOG_AREA_DENSFROMF, "FileWritable::writeAndReadAll");
  do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, ((std::string)"writeAndReadAll sizesStr: '" + sizesStr).c_str());

  // we can clear here X^2 !
  Xsq.clear();
     

  assert(eigVecHOMO != NULL);
  assert(eigVecLUMO != NULL);

  // we can clear here X^2 ! it is not needed anymore
  Xsq.clear();

  // check if we passed iter_for_homo iteration and saved the matrix
  if(info.total_it >= iter_for_homo)
    {
      // reading X_homo matrix from the bin file
      Util::TimeMeter X_homo_read;
      X_homo.readFromFile();
      X_homo_read.print(LOG_AREA_DENSFROMF, "reading X matrix (for homo) using readFromFile");


      Util::TimeMeter homo_total_time; // total time for homo
	
      MatrixType DXi(X);

      // multiplying D*Xi
      Util::TimeMeter DX_mult_time_homo;
      DXi.mult_force_symm(X_homo); // D*Xi
      DX_mult_time_homo.print(LOG_AREA_DENSFROMF, "computing D*X (for homo)");
      DX_mult_time_homo_stop = DX_mult_time_homo.get_elapsed_wall_seconds();

      // get HOMO matrix
      // note: we may need DXi for computing lumo, do not overwrite it
      MatrixType Zh(X);
      Zh.subtract(DXi); // D-DXi
      Util::TimeMeter homo_solver_time;
      compute_eigenvector(Zh, eigVecHOMO, iter_for_homo, true);
      homo_solver_time.print(LOG_AREA_DENSFROMF, "compute_eigenvector()");
      homo_solver_time_stop = homo_solver_time.get_elapsed_wall_seconds();

      info.Iterations[iter_for_homo].homo_eig_solver_time = homo_solver_time_stop; // note: here is included just time for compute_eigenvector()
      info.Iterations[iter_for_homo].DX_mult_homo_time = DX_mult_time_homo_stop;
      Zh.clear();

      homo_total_time.print(LOG_AREA_DENSFROMF, "computing homo eigenvector");
      homo_total_time_stop = homo_total_time.get_elapsed_wall_seconds();
      info.Iterations[iter_for_homo].orbital_homo_time = homo_total_time_stop;
	 
	 
      // if we are computing both homo and lumo in the same iteration
      if(iter_for_homo == iter_for_lumo)
	{
	  Util::TimeMeter lumo_total_time;

	  do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Reuse matrix D*X_i for lumo computations"); // no matrix read

	  // get LUMO matrix
	  DXi.subtract(X_homo);
	  DXi.mult_scalar(-1); // Xi-DXi

	  Util::TimeMeter lumo_solver_time;
	  compute_eigenvector(DXi, eigVecLUMO, iter_for_lumo, false);
	  lumo_solver_time.print(LOG_AREA_DENSFROMF, "compute_eigenvector()");
	  lumo_solver_time_stop = lumo_solver_time.get_elapsed_wall_seconds();


	  lumo_total_time.print(LOG_AREA_DENSFROMF, "computing lumo eigenvector");
	  lumo_total_time_stop = lumo_total_time.get_elapsed_wall_seconds();

	  info.Iterations[iter_for_lumo].DX_mult_lumo_time = 0; // we reuse DX matrix
	  info.Iterations[iter_for_lumo].lumo_eig_solver_time = lumo_solver_time_stop;  // note: here is included just time for compute_eigenvector()
	  info.Iterations[iter_for_lumo].orbital_lumo_time = lumo_total_time_stop;
	} // LUMO
    } // HOMO

  X_homo.clear();

  // check if we passed iter_for_lumo iteration and saved the matrix, and that it was not the same iteration
  // as iter_for_homo
  if(info.total_it >= iter_for_lumo && iter_for_homo != iter_for_lumo)
    {
      Util::TimeMeter X_lumo_read;
      X_lumo.readFromFile();
      X_lumo_read.print(LOG_AREA_DENSFROMF, "reading X matrix (for lumo) using readFromFile");
       
      Util::TimeMeter lumo_total_time;

      MatrixType DXi(X); // D
      
      Util::TimeMeter DX_mult_time_lumo;
      DXi.mult_force_symm(X_lumo); // D*Xi
      DX_mult_time_lumo.print(LOG_AREA_DENSFROMF, "computing D*X (for lumo)");
      DX_mult_time_lumo_stop = DX_mult_time_lumo.get_elapsed_wall_seconds();

      // get LUMO matrix
      DXi.subtract(X_lumo);
      DXi.mult_scalar(-1); // Xi-DXi

      Util::TimeMeter lumo_solver_time;
      compute_eigenvector(DXi, eigVecLUMO, iter_for_lumo, false);
      lumo_solver_time.print(LOG_AREA_DENSFROMF, "compute_eigenvector()");
      lumo_solver_time_stop = lumo_solver_time.get_elapsed_wall_seconds();
      info.Iterations[iter_for_lumo].lumo_eig_solver_time = lumo_solver_time_stop;
      info.Iterations[iter_for_lumo].DX_mult_lumo_time = DX_mult_time_lumo_stop;

      lumo_total_time.print(LOG_AREA_DENSFROMF, "computing lumo eigenvector");
      lumo_total_time_stop = lumo_total_time.get_elapsed_wall_seconds();
      info.Iterations[iter_for_lumo].orbital_lumo_time = lumo_total_time_stop;

      X_lumo.clear();
    } // LUMO 

     
  output_current_memory_usage(LOG_AREA_DENSFROMF, "After computing eigenvectors:");
}
#endif // USE_CHUNKS_AND_TASKS




template<typename MatrixType>
void PurificationGeneral<MatrixType>::determine_iteration_for_eigenvectors()
{
  if(eigenvectors_method == EIG_SQUARE_INT || eigenvectors_method == EIG_PROJECTION_INT)
    {
      get_iterations_for_lumo_and_homo(iter_for_lumo, iter_for_homo);
      do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Eigenvector for HOMO will be computed on the iteration %d. ", iter_for_homo);
      do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Eigenvector for LUMO will be computed on the iteration %d. ", iter_for_lumo);
    }
  else
    throw "Error in determine_iteration_for_eigenvectors: unknown method for computing eigenvectors.";
}


/*************** get_iterations_for_lumo_and_homo  ******************//**
 *  Find the best iterations for computing eigenvectors.
 *  The best iteration defined by both a small error in the eigenvector
 *  and a small number of iterations of an iterative method.
 *********************************************************************/
template<typename MatrixType>
void PurificationGeneral<MatrixType>::get_iterations_for_lumo_and_homo(int &chosen_iter_lumo,
								       int &chosen_iter_homo)
{
  real TOL_ERR = get_epsilon();
  int maxiter = maxit;
  // Inner bounds (from the previous SCF cycle {i-1}, expanded
  // with norm of F_i - F_{i-1}).
  real homo0 = 1-homo_bounds.upp(); // inner bounds
  real lumo0 = lumo_bounds.upp();
  real homo_out = 1-homo_bounds.low(); // outer bounds
  real lumo_out = lumo_bounds.low();

  real homo = homo0, lumo = lumo0;

  good_iterations_homo.clear();
  good_iterations_lumo.clear();


  chosen_iter_lumo = -1;
  chosen_iter_homo = -1;

  // (derivative is always positive (we are using approximation of the step function which is monotonically growing) - not true for the acceleration!)
  real dx1, dx2, dx1max = get_min_double(), dx2max = get_min_double();

  for(int i = 1; i <= maxiter; ++i)
    {
      
      // get derivative of the function at homo and lumo
      // find in which iteration the derivative is the largest
      dx1 = compute_derivative(i, homo0);
      dx2 = compute_derivative(i, lumo0);

      homo = apply_poly(i, homo); // apply POLY[i] on homo
      if(homo == -1) throw "get_iterations_for_homo:error in apply_poly";
      lumo = apply_poly(i, lumo); // apply POLY[i] on lumo
      if(lumo == -1) throw "get_iterations_for_homo:error in apply_poly";

      // find all possible iterations for computing eigenvectors
      if(eigenvectors_method == EIG_SQUARE_INT)
	{
	  // choice of shift restricts number of possible iterations
	  if(homo > (homo_out + lumo)/2)
	    {
	      good_iterations_homo.push_back(i);
	    }
	  if(lumo < (homo + lumo_out)/2)
	    {
	      good_iterations_lumo.push_back(i);
	    }
	}
      else if(eigenvectors_method == EIG_PROJECTION_INT)
	{
	  // all iterations are good
	  good_iterations_homo.push_back(i);
	  good_iterations_lumo.push_back(i);
	}
      else throw "Error in get_iterations_for_lumo_and_homo() : wrong eigenvectors_method value.";

      homo_out = apply_poly(i, homo_out); // apply POLY[i] on homo_out
      lumo_out = apply_poly(i, lumo_out); // apply POLY[i] on lumo_out

      // FIND MAXIMUM OF THE DERIVATIVE OVER ALL ITERATIONS
      if(dx1max < dx1)  dx1max = dx1;
      if(dx2max < dx2)  dx2max = dx2;
    }

  // now we know the value of the largest derivative 
  
  real PART = 0.5; // 50%
  real DX1_BOUND = PART*dx1max;
  real DX2_BOUND = PART*dx2max;
  bool found_homo, found_lumo;
  
  for(int i = maxiter; i >= 1; --i)
    {
      dx1 = compute_derivative(i, homo0);
      dx2 = compute_derivative(i, lumo0); 

      // check if it is an allowed iteration
      found_homo = std::binary_search(good_iterations_homo.begin(), good_iterations_homo.end(), i); // we have sorted vector
      found_lumo = std::binary_search(good_iterations_lumo.begin(), good_iterations_lumo.end(), i); // we have sorted vector

      if(found_homo && dx1 >= DX1_BOUND && chosen_iter_homo == -1) chosen_iter_homo = i;
      if(found_lumo && dx2 >= DX2_BOUND && chosen_iter_lumo == -1) chosen_iter_lumo = i;

      if(chosen_iter_lumo != -1 && chosen_iter_homo != -1) break;
    }

  if(chosen_iter_homo == -1 || chosen_iter_lumo == -1)
    throw "Error in get_iterations_for_lumo_and_homo() : something went wrong, cannot choose iteration to compute HOMO or LUMO eigenvector.";
}






#ifdef USE_CHUNKS_AND_TASKS
template<typename MatrixType>
void PurificationGeneral<MatrixType>::compute_eigenvector(MatrixType const &M, VectorType* eigVecHOMOorLUMO, int it, bool is_homo)
{throw "compute_eigenvector() is not implemented for CHTMatrix.";}
#else
template<typename MatrixType>
void PurificationGeneral<MatrixType>::compute_eigenvector(MatrixType const &M, VectorType* eigVecHOMOorLUMO, int it, bool is_homo)
{
  assert(eigVecHOMOorLUMO != NULL);
  real acc_eigv = eigensolver_accuracy;

#ifdef DEBUG_PURI_OUTPUT
  do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Starting compute_eigenvector() in iteration %d", it);
#endif


  vector<real> eigValTmp(number_of_eigenvalues); // here will be computed eigenvalues of M
  mat::SizesAndBlocks rows;


  X.get_ref_to_matrix().getRows(rows);
  /* 
     Apparently the std::vector constructor calls the copy constructor which is not allowed if the data structure of VectorType is not set. 
     In VectorGeneral class were added a new constructor receiving data structure.
   */
  vector<VectorType> eigVec(number_of_eigenvalues, VectorType(rows)); // here will be computed eigenvectors
  if(use_prev_vector_as_initial_guess) // copy vector from previous SCF cycle to be an initial guess
    {
      if(is_homo)
	eigVec[0] = *eigVecHOMO;
      else
	eigVec[0] = *eigVecLUMO;
    }


  vector<int> num_iter_solver(number_of_eigenvalues, -1); // number of iterations 

  Util::TimeMeter computeEigenvectors_time;
  // run non-deflated version
  int eig_num = 1;
  MatrixType::computeEigenvectors(M, acc_eigv,  eigValTmp, eigVec, eig_num, use_prev_vector_as_initial_guess,  
  				  eigenvectors_iterative_method_str, num_iter_solver, 
				  eigensolver_maxiter);
  computeEigenvectors_time.print(LOG_AREA_DENSFROMF, "eigensolver");

  if(num_iter_solver.empty())
    throw "Error in compute_eigenvector() : (num_iter_solver.empty())";


  // initialize
  bool is_homo2 = false, is_lumo2 = false;
  bool is_homo_tmp = false, is_lumo_tmp = false;

  if(num_iter_solver[0] == eigensolver_maxiter)
    do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Eigensolver did not converge within maxiter = %d iterations.", eigensolver_maxiter);
  else
    {
      do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Number of eigensolver iterations is %d", num_iter_solver[0]);

      is_homo_tmp = is_homo, is_lumo_tmp = !is_homo;
      real eigVal;
      get_interval_with_lambda(eigVal, eigVec[0], is_homo_tmp, is_lumo_tmp, it); // compute also the corresponding eigenvalue of F
      if(is_homo_tmp)
	{
	  really_good_iterations_homo.push_back(it);
	  *eigVecHOMO = eigVec[0];
	  info.homo_eigenvector_is_computed = true;
	  eigValHOMO = eigVal;
	}
      else if(is_lumo_tmp)
	{
	  really_good_iterations_lumo.push_back(it); // iterations where we computed lumo (for any reason)
	  *eigVecLUMO = eigVec[0];
	  info.lumo_eigenvector_is_computed = true;
	  eigValLUMO = eigVal;
	}
      else 
	do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Error in compute_eigenvector: wrong eigenvalue ");
    }
}

#endif // USE_CHUNKS_AND_TASKS


      
template<typename MatrixType>
void PurificationGeneral<MatrixType>::get_eigenvalue_of_F_from_eigv_of_Xi(real &eigVal, VectorType &eigVec)
{
  F.readFromFile();
  real approx_eig = MatrixType::compute_rayleigh_quotient(F, &eigVec);
  F.writeToFile();
  eigVal = approx_eig;
}



template<typename MatrixType>
void PurificationGeneral<MatrixType>::get_interval_with_lambda(real &eigVal, VectorType &eigVec, bool &is_homo, bool &is_lumo, const int iter)
{
  assert(is_homo || is_lumo);

  bool is_homo_init = is_homo;
  bool is_lumo_init = is_lumo;

  /* COMPUTE EIGENVALUE OF F CORRESPONDING TO THE COMPUTED EIGENVECTOR USING RAYLEIGH QUOTIENT. */

  /*
    Note: For the square method we compute eigenvalues in the current
    iteration during the purification. The bounds lumo_bounds and
    homo_bounds are changing in every iteration according to the
    polynomial (without expansion by tau). Thus we should use this
    bounds if square method is used.  However, for the projection
    method we should used bounds which are saved into the info object.
   */
  real low_lumo_F_bound, low_homo_F_bound;
  real upp_lumo_F_bound, upp_homo_F_bound;

  if(eigenvectors_method == EIG_SQUARE_INT)
    // for square method use bounds from the previous SCF cycle (i.e. bounds expanded with norm ||F_i-F_{i-1}||)
    {
      low_lumo_F_bound = lumo_bounds_F.low();
      upp_lumo_F_bound = lumo_bounds_F.upp();
      low_homo_F_bound = homo_bounds_F.low();
      upp_homo_F_bound = homo_bounds_F.upp();
    }
  else if(eigenvectors_method == EIG_PROJECTION_INT) 
    // for projection method we can use new bounds computed in this SCF cycle
    {
      low_lumo_F_bound = info.lumo_estim_low_F;
      upp_lumo_F_bound = info.lumo_estim_upp_F;
      low_homo_F_bound = info.homo_estim_low_F;
      upp_homo_F_bound = info.homo_estim_upp_F;
    }
  else
    throw "Error in get_interval_with_lambda() : unexpected eigenvectors_method value.";

#ifdef DEBUG_PURI_OUTPUT
  do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Check Rayleigh quotient...");
#endif

  F.readFromFile();
  real approx_eig = MatrixType::compute_rayleigh_quotient(F, &eigVec);
  F.writeToFile();

  eigVal = approx_eig;

  real flex_tolerance = THRESHOLD_EIG_TOLERANCE;

  // it is HOMO
  if(approx_eig <= upp_homo_F_bound + flex_tolerance && approx_eig >= low_homo_F_bound - flex_tolerance)
    {
      is_homo = true; is_lumo = false;
#ifdef DEBUG_PURI_OUTPUT
      do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Computed HOMO eigenvalue of F is %lf, "
		"HOMO bounds are  [ %lf , %lf ]", approx_eig, low_homo_F_bound, upp_homo_F_bound);
#endif
      iter_for_homo = iter; // We already computed homo in this iteration (in case we thought that it was lumo)
      // Do we want to recompute lumo in the next iteration?
      if(is_lumo_init && eigenvectors_method == EIG_SQUARE_INT && try_eigv_on_next_iteration_if_fail) 
	{
	  iter_for_lumo = iter_for_lumo+1;
	  do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "We will try to compute LUMO in the next iteration.");
	}
    }
  // it is LUMO
  else if(approx_eig <= upp_lumo_F_bound + flex_tolerance && approx_eig >= low_lumo_F_bound - flex_tolerance)
    {
      is_homo = false; is_lumo = true;
#ifdef DEBUG_PURI_OUTPUT
      do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Computed LUMO eigenvalue of F is %lf, "
		"LUMO interval [ %lf , %lf ]", approx_eig, low_lumo_F_bound, upp_lumo_F_bound);
#endif
      iter_for_lumo = iter; // We already computed lumo in this iteration (in case we thought that it was homo)
      // Do we want to recompute homo in the next iteration?
      if(is_homo_init && eigenvectors_method == EIG_SQUARE_INT && try_eigv_on_next_iteration_if_fail) 
	{
	  iter_for_homo = iter_for_homo+1;
	  do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "We will try to compute HOMO in the next iteration.");
	}
    }
  else
    {
      is_homo = false; is_lumo = false;
#ifdef DEBUG_PURI_OUTPUT
      do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Eigenvalue is outside of both intervals for homo and lumo.");
      do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Eigenvalue is %lf,  HOMO interval [ %lf , %lf ], LUMO interval [ %lf , %lf ]", 
		approx_eig, low_homo_F_bound, upp_homo_F_bound, low_lumo_F_bound, upp_lumo_F_bound);
#endif
      // Do we want to recompute homo (or lumo) in the next iteration?
      // We will try to compute HOMO, however, it can be LUMO.
      // If it will be LUMO, we save it as computed LUMO and continue with computing HOMO in the next iteration.
      if(eigenvectors_method == EIG_SQUARE_INT && try_eigv_on_next_iteration_if_fail) 
	{
	  iter_for_homo = iter_for_homo+1;
	  do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "We will try to compute HOMO (or LUMO) in the next iteration.");
	}
    }

}





/******************************************************************/
/*********************** MATLAB FUNCTIONS *************************/
/******************************************************************/

template<typename MatrixType>
void PurificationGeneral<MatrixType>::gen_matlab_file_norm_diff(const char *filename) const
{
  std::ofstream f;
  f.open (filename, std::ios::out);
  if (!f.is_open())
    {
      do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Error: cannot open file");
      return;
    }

  int it = info.total_it;
  // save POLY
  f << "POLY = [";
  for(int i = 0; i <= it; ++i)
    {
      f << VecPoly[i] << " ";
    }
  f << "];" << std::endl;
  
  // choose norm
  if( normPuriStopCrit == mat::euclNorm )
    {
      f << "X_norm = [";
      for(int i = 0; i <= it; ++i) // including 0 iteration = original matrix X
	{
	  f << info.Iterations[i].XmX2_eucl << " ";
	}
      f << "];" << std::endl;
      f << " norm_letter = '2';" << std::endl;
    }
  else if( normPuriStopCrit == mat::frobNorm )
    {
      f << "X_norm = [";
      for(int i = 0; i <= it; ++i) // including 0 iteration = original matrix X
	{
	  f << info.Iterations[i].XmX2_fro_norm << " ";
	}
      f << "];" << std::endl;
      f << " norm_letter = 'F';" << std::endl;
    }
  else if( normPuriStopCrit == mat::mixedNorm )
    {
      f << "X_norm = [";
      for(int i = 0; i <= it; ++i) // including 0 iteration = original matrix X
	{
	  f << info.Iterations[i].XmX2_mixed_norm << " ";
	}
      f << "];" << std::endl;
      f << " norm_letter = 'M';" << std::endl;
    }
  else throw "Wrong norm in PurificationGeneral::gen_matlab_file_norm_diff()";
  
  f << "stop_iteration = " << it - info.additional_iterations << ";" << std::endl;
  f << "it = " << it << ";" << std::endl;
  f << "plot_props = {'LineWidth', 2, 'MarkerSize', 8};" << std::endl; 
  f << "fighandle = figure; clf;" << std::endl;
  f << "MARKER = ['o', '+'];" << std::endl;
  f << "semilogy(0:stop_iteration, X_norm(1:stop_iteration+1), '-b', plot_props{:});" << std::endl; 
  f << "hold on" << std::endl;
  f << "for i = 1:stop_iteration+1" << std::endl;
  f << " if POLY(i) == 1" << std::endl;
  f << "  h1 = semilogy(i-1, X_norm(i), [MARKER((POLY(i) == 1) + 1) 'b'], plot_props{:});" << std::endl;
  f << " else" << std::endl;
  f << "  h2 = semilogy(i-1, X_norm(i), [MARKER((POLY(i) == 1) + 1) 'b'], plot_props{:});" << std::endl;
  f << " end" << std::endl;
  f << "end" << std::endl;
  f << "if stop_iteration ~= it" << std::endl;     
  f << "h3 = semilogy(stop_iteration+1:it, X_norm(stop_iteration+2:it+1), '-.vr', plot_props{:});"  << std::endl;
  f << "semilogy(stop_iteration:stop_iteration+1, X_norm(stop_iteration+1:stop_iteration+2), '-.r', plot_props{:});" << std::endl;
  f << "legend([h1 h2 h3],{'$x^2$', '$2x-x^2$', 'After stop'}, 'Interpreter','latex', 'Location','SouthWest');" << std::endl; 
  f << "else" << std::endl;
  f << "legend([h1 h2],{'$x^2$', '$2x-x^2$'}, 'Interpreter','latex', 'Location','SouthWest');" << std::endl; 
  f << "end" << std::endl;
  f << "xlabel('Iteration SP2', 'Interpreter','latex');" << std::endl; 
  f << "ylabel({['$\\|X_i-X_i^2\\|_{' norm_letter '}$']},'interpreter','latex');" << std::endl;
  f << "grid on" << std::endl;
  f << "set(gca, 'XMinorGrid','off', 'YMinorGrid','off', 'GridAlpha',0.6, 'GridLineStyle', '--')" << std::endl;
  f << "set(gca,'FontSize',20);" << std::endl;
  f << "xlim([0 it]);" << std::endl;
  f << "ylim([-inf inf]);" << std::endl;
  f << "set(gca,'XTick',[0 5:5:it]);" << std::endl;
  f << "a = 16; S = logspace(-a, 1, a+2);" << std::endl;
  f << "set(gca,'YTick',S(1:2:end));" << std::endl;

  f << "hold off" << std::endl;

  f << "% print( fighandle, '-depsc2', 'norm_diff.eps');" << std::endl;

  f.close();
}


 
template<typename MatrixType>
void PurificationGeneral<MatrixType>::gen_matlab_file_threshold(const char *filename) const
{
  std::ofstream f;
  f.open (filename, std::ios::out);
  if (!f.is_open())
    {
      do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Error: cannot open file");
      return;
    }

  int it = info.total_it;
  f << "Thresh = [";
   
  for(int i = 0; i <= it; ++i)
    {
      f << info.Iterations[i].threshold_X << " ";
    }
  f << "];" << std::endl;

  f << "stop_iteration = " << it - info.additional_iterations << ";" << std::endl;
  f << "it = " << it << ";" << std::endl;
  f << "plot_props = {'LineWidth', 2, 'MarkerSize', 8};" << std::endl; 
  f << "fighandle = figure; clf;" << std::endl;
  f << "semilogy(0:stop_iteration, Thresh(1:stop_iteration+1), '-vb', plot_props{:});" << std::endl;
  f << "hold on" << std::endl;
  f << "if stop_iteration ~= it" << std::endl;
  f << "semilogy(stop_iteration+1:it, Thresh(stop_iteration+2:it+1), '-^r', plot_props{:});"  << std::endl;
  f << "semilogy(stop_iteration:stop_iteration+1, Thresh(stop_iteration+1:stop_iteration+2), '-r', plot_props{:});" << std::endl;
  f << "legend('before stop', 'after stop', 'Location','NorthWest');" << std::endl; 
  f << "end" << std::endl;
  f << "grid on" << std::endl;
  f << "set(gca, 'XMinorGrid','off', 'YMinorGrid','off', 'GridAlpha',0.6, 'GridLineStyle', '--')" << std::endl;
  f << "set(gca,'FontSize',20);" << std::endl;
  f << "xlim([0 it]);" << std::endl;
  f << "ylim([-inf inf]);" << std::endl;
  f << "set(gca,'XTick',[0 5:5:it]);" << std::endl;
  f << "hold off" << std::endl;
  f << "xlabel('Iteration SP2', 'interpreter','latex');" << std::endl; 
  f << "ylabel('Threshold value', 'interpreter','latex');" << std::endl;
  f << "% print( fighandle, '-depsc2', 'threshold.eps');" << std::endl;

  f.close();
}



template<typename MatrixType>
void PurificationGeneral<MatrixType>::gen_matlab_file_nnz(const char *filename) const
{
  std::ofstream f;
  f.open (filename, std::ios::out);
  if (!f.is_open())
    {
      do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Error: cannot open file");                   
      return;
    }

  int it = info.total_it;
  f << "NNZ_X = [";
   
  for(int i = 0; i <= it; ++i)
    {
      f << info.Iterations[i].NNZ_X << " ";
    }
  f << "];" << std::endl;

  f << "NNZ_X2 = [";
   
  for(int i = 0; i <= it; ++i)
    {
      f << info.Iterations[i].NNZ_X2 << " ";
    }
  f << "];" << std::endl;



  f << "stop_iteration = " << it - info.additional_iterations << ";" << std::endl;
  f << "it = " << it << ";" << std::endl;
  f << "plot_props = {'LineWidth', 2, 'MarkerSize', 8};" << std::endl; 
  f << "fighandle = figure; clf;" << std::endl;
  f << "h2 = plot(0:stop_iteration, NNZ_X(1:stop_iteration+1), '-ob', plot_props{:});" << std::endl;
  f << "hold on" << std::endl; 
  f << "h1 = plot(0:stop_iteration, NNZ_X2(1:stop_iteration+1), '-vm', plot_props{:});" << std::endl;
  f << "if stop_iteration ~= it" << std::endl;
  f << "plot(stop_iteration+1:it, NNZ_X(stop_iteration+2:it+1), '-vr', plot_props{:});"  << std::endl;
  f << "plot(stop_iteration+1:it, NNZ_X2(stop_iteration+2:it+1), '-*r', plot_props{:});"  << std::endl;
  f << "plot(stop_iteration:stop_iteration+1, NNZ_X(stop_iteration+1:stop_iteration+2), '-r', plot_props{:});" << std::endl;
  f << "plot(stop_iteration:stop_iteration+1, NNZ_X2(stop_iteration+1:stop_iteration+2), '-r', plot_props{:});" << std::endl;
  f << "end" << std::endl;
  f << "legend([h1, h2], {'$X^2$', '$X$'},  'interpreter','latex');" << std::endl; 
  f << "grid on" << std::endl;
  f << "set(gca, 'XMinorGrid','off', 'YMinorGrid','off', 'GridAlpha',0.6, 'GridLineStyle', '--')" << std::endl;
  f << "set(gca,'FontSize',20);" << std::endl;
  f << "xlim([0 it]);" << std::endl;
  f << "ylim([0 inf]);" << std::endl;
  f << "set(gca,'XTick',[0 5:5:it]);" << std::endl;
  f << "hold off" << std::endl;
  f << "xlabel('Iteration SP2', 'interpreter','latex');" << std::endl; 
  f << "ylabel('NNZ [\\%]', 'interpreter','latex');" << std::endl;
  f << "% print( fighandle, '-depsc2', 'nnz.eps');" << std::endl;

  f.close();
}   


template<typename MatrixType>
void PurificationGeneral<MatrixType>::gen_matlab_file_cond_num(const char *filename) const
{
  std::ofstream f;
  f.open (filename, std::ios::out);
  if (!f.is_open())
    {
      do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Error: cannot open file");
      return;
    }

  int it = info.total_it;
  f << "in= [";
   
  for(int i = 0; i <= it; ++i)
    {
      f << 1 - info.Iterations[i].homo_bound_upp - info.Iterations[i].lumo_bound_upp << " ";
    }
  f << "];" << std::endl;

  f << "stop_iteration = " << it - info.additional_iterations << ";" << std::endl;
  f << "plot_props = {'LineWidth', 2, 'MarkerSize', 8};" << std::endl; 
  f << "fighandle = figure; clf;" << std::endl;
  f << "plot(0:stop_iteration, 1./in(1:stop_iteration+1), '-*r', plot_props{:});" << std::endl;
  f << "grid on" << std::endl;
  f << "set(gca, 'XMinorGrid','off', 'YMinorGrid','off', 'GridAlpha',0.6, 'GridLineStyle', '--')" << std::endl;
  f << "set(gca,'FontSize',20);" << std::endl;
  f << "xlim([0 it]);" << std::endl;
  f << "ylim([-inf inf]);" << std::endl;
  f << "set(gca,'XTick',[0 5:5:it]);" << std::endl;
  f << "hold off" << std::endl;
  f << "xlabel('Iteration SP2', 'interpreter','latex');" << std::endl; 
  f << "ylabel('$\\kappa$', 'interpreter','latex');" << std::endl;
  f << "% print( fighandle, '-depsc2', 'cond.eps');" << std::endl;

  f.close();
} 



template<typename MatrixType>
void PurificationGeneral<MatrixType>::gen_matlab_file_eigs(const char *filename) const
{
  std::ofstream f;
  f.open (filename, std::ios::out);
  if (!f.is_open())
    {
      do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Error: cannot open file");
      return;
    }

  int it = info.total_it;
  f << "homo_low= [";
   
  for(int i = 0; i <= it; ++i)
    {
      f << info.Iterations[i].homo_bound_low << " ";
    }
  f << "];" << std::endl;

  f << "homo_upp= [";
   
  for(int i = 0; i <= it; ++i)
    {
      f << info.Iterations[i].homo_bound_upp << " ";
    }
  f << "];" << std::endl;

  f << "lumo_low= [";
   
  for(int i = 0; i <= it; ++i)
    {
      f << info.Iterations[i].lumo_bound_low << " ";
    }
  f << "];" << std::endl;

  f << "lumo_upp= [";
   
  for(int i = 0; i <= it; ++i)
    {
      f << info.Iterations[i].lumo_bound_upp << " ";
    }
  f << "];" << std::endl;



  f << "stop_iteration = " << it - info.additional_iterations << ";" << std::endl;
  f << "plot_props = {'LineWidth', 2, 'MarkerSize', 8};" << std::endl; 
  f << "fighandle = figure; clf;" << std::endl;
  f << "semilogy(0:stop_iteration, homo_upp(1:stop_iteration+1), '-^b', plot_props{:});" << std::endl;
  f << "hold on" << std::endl; 
  f << "semilogy(0:stop_iteration, homo_low(1:stop_iteration+1), '-vb', plot_props{:});" << std::endl;
  f << "semilogy(0:stop_iteration, lumo_low(1:stop_iteration+1), '-vr', plot_props{:});" << std::endl;
  f << "semilogy(0:stop_iteration, lumo_upp(1:stop_iteration+1), '-^r', plot_props{:});" << std::endl;
  f << "grid on" << std::endl;
  f << "set(gca, 'XMinorGrid','off', 'YMinorGrid','off', 'GridAlpha',0.6, 'GridLineStyle', '--')" << std::endl;
  f << "set(gca,'FontSize',20);" << std::endl;
  f << "xlim([0 stop_iteration]);" << std::endl;
  f << "set(gca,'XTick',[0 5:5:stop_iteration]);" << std::endl;
  f << "ylim([-inf inf]);" << std::endl;
  f << "hold off" << std::endl;
  f << "xlabel('Iteration SP2', 'interpreter','latex');" << std::endl; 
  f << "ylabel('Homo and lumo bounds', 'interpreter','latex');" << std::endl;
  f << "% print( fighandle, '-depsc2', 'eigs.eps');" << std::endl;

  f.close();
} 



template<typename MatrixType>
void PurificationGeneral<MatrixType>::gen_matlab_file_time(const char *filename) const
{
  std::ofstream f;
  f.open (filename, std::ios::out);
  if (!f.is_open())
    {
      do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Error: cannot open file");
      return;
    }

  int it = info.total_it;

  f << "time_total = [";   
  for(int i = 0; i <= it; ++i)
    {
      f << std::setprecision(16) << info.Iterations[i].total_time << " ";
    }
  f << "];" << std::endl;

  f << "time_square = [";   
  for(int i = 0; i <= it; ++i)
    {
      f << std::setprecision(16) << info.Iterations[i].Xsquare_time << " ";
    }
  f << "];" << std::endl;

  f << "time_trunc = [";   
  for(int i = 0; i <= it; ++i)
    {
      f << std::setprecision(16) << info.Iterations[i].trunc_time << " ";
    }
  f << "];" << std::endl;

  if(info.compute_eigenvectors_in_this_SCF_cycle)
    {
      f << "time_eigenvectors_homo = [";   
      for(int i = 0; i <= it; ++i)
	{
	  f << std::setprecision(16) << info.Iterations[i].orbital_homo_time << " ";
	}
      f << "];" << std::endl;
      f << "time_eigenvectors_lumo = [";   
      for(int i = 0; i <= it; ++i)
	{
	  f << std::setprecision(16) << info.Iterations[i].orbital_lumo_time << " ";
	}
      f << "];" << std::endl;

      f << "time_solver_homo = [";   
      for(int i = 0; i <= it; ++i)
	{
	  f << std::setprecision(16) << info.Iterations[i].homo_eig_solver_time << " ";
	}
      f << "];" << std::endl;
      f << "time_solver_lumo = [";   
      for(int i = 0; i <= it; ++i)
	{
	  f << std::setprecision(16) << info.Iterations[i].lumo_eig_solver_time << " ";
	}
      f << "];" << std::endl; 


      if(eigenvectors_method == EIG_SQUARE_INT)
	{
	  // time for X^2, truncation, eigensolver for homo and lumo, additional time for homo and lumo and other time
	  f << "X = [time_square; time_trunc; time_solver_homo; time_solver_lumo; time_eigenvectors_homo-time_solver_homo;"
	    " time_eigenvectors_lumo-time_solver_lumo; "
	    " time_total - time_square - time_trunc - time_eigenvectors_homo - time_eigenvectors_lumo];" << std::endl;
	}
      else
	{
	  f << "time_DX_homo = [";   
	  for(int i = 0; i <= it; ++i)
	    {
	      f << std::setprecision(16) << info.Iterations[i].DX_mult_homo_time << " ";
	    }
	  f << "];" << std::endl;
	  f << "time_DX_lumo = [";   
	  for(int i = 0; i <= it; ++i)
	    {
	      f << std::setprecision(16) << info.Iterations[i].DX_mult_lumo_time << " ";
	    }
	  f << "];" << std::endl;
	  
	  // for projection total time of the iteration does not include computation of homo and lumo
	  // time for X^2, eigensolver for homo and lumo, DX multiplication
	  f << "X = [time_square; time_trunc; time_solver_homo; time_solver_lumo; time_DX_homo; time_DX_lumo;"
	    " time_eigenvectors_homo - time_DX_homo - time_solver_homo; time_eigenvectors_lumo - time_DX_lumo - time_solver_lumo;"
	    " time_total - time_square - time_trunc];" << std::endl;
	}

    }
  else
    {
      f << "X = [time_square; time_trunc; time_total - time_square - time_trunc];" << std::endl;
    }

  f << "it = " << it << ";" << std::endl;
  f << "xtick = 0:it;" << std::endl;
  f << "fighandle = figure; clf;" << std::endl;
  f << "b=bar(xtick, X', 'stacked');" << std::endl;
  f << "axis tight;" << std::endl;
  f << "grid on" << std::endl;
  f << "set(gca, 'XMinorGrid','off', 'YMinorGrid','off', 'GridAlpha',0.6, 'GridLineStyle', '--')" << std::endl;
  f << "set(gca,'FontSize',20);" << std::endl;
  f << "xlim([0 it]);" << std::endl;
  f << "set(gca,'XTick',[0 5:5:it]);" << std::endl;
  f << "ylim([-inf inf]);" << std::endl;
  f << "xlabel('Iteration SP2', 'interpreter','latex');" << std::endl; 
  f << "ylabel('Time [s]', 'interpreter','latex');" << std::endl;
  if(info.compute_eigenvectors_in_this_SCF_cycle)
/*
Legend with matlab's bar appear with default settings in reverse or order. Thus we force it to follow the order of the bar manually.
*/
    if(eigenvectors_method == EIG_SQUARE_INT)
      f << "legend(flipud(b(:)), {'other', 'lumo other', 'homo other', 'lumo solver', 'homo solver', 'truncation', '$X^2$'}, 'interpreter','latex');" << std::endl; 
    else
      f << "legend(flipud(b(:)), {'other', 'lumo other', 'homo other', 'DX (lumo)', 'DX (homo)', 'lumo solver', 'homo solver', 'truncation', '$X^2$'}, 'interpreter','latex');" << std::endl;
  else
    f << "legend(flipud(b(:)), {'other', 'truncation', '$X^2$'}, 'interpreter','latex');" << std::endl;
  f << "% print( fighandle, '-depsc2', 'time.eps');" << std::endl;

  f.close();
}



template<typename MatrixType>
void PurificationGeneral<MatrixType>::gen_matlab_file_which_eig_computed(const char *filename) const
{
  if(!compute_eigenvectors_in_this_SCF_cycle) return;

  std::ofstream f;
  f.open (filename, std::ios::out);
  if (!f.is_open())
    {
      do_output(LOG_CAT_INFO, LOG_AREA_DENSFROMF, "Error: cannot open file");
      return;
    }

  // iterations where Ergo thinks it will compute homo
  f << "good_iterations_homo = [";   
  for(int i = 0; i < (int)good_iterations_homo.size(); ++i)
    {
      f << good_iterations_homo[i] << " ";
    }
  f << "];" << std::endl;

 // iterations where Ergo compute homo
  f << "really_good_iterations_homo = [";   
  for(int i = 0; i < (int)really_good_iterations_homo.size(); ++i)
    {
      f << really_good_iterations_homo[i] << " ";
    }
  f << "];" << std::endl;


  // iterations where Ergo thinks it will compute lumo
  f << "good_iterations_lumo = [";   
  for(int i = 0; i < (int)good_iterations_lumo.size(); ++i)
    {
      f << good_iterations_lumo[i] << " ";
    }
  f << "];" << std::endl;

 // iterations where Ergo compute lumo
  f << "really_good_iterations_lumo = [";   
  for(int i = 0; i < (int)really_good_iterations_lumo.size(); ++i)
    {
      f << really_good_iterations_lumo[i] << " ";
    }
  f << "];" << std::endl;

  f << "total_it = max(good_iterations_homo(end), good_iterations_lumo(end));" << std::endl;

  f << "plot_props = {'LineWidth', 2, 'MarkerSize', 8}; " << std::endl;
  f << "yhomo = 2;" << std::endl;
  f << "ylumo = 1;" << std::endl;
  f << "fighandle = figure; clf; " << std::endl;
  f << "h1 = plot(good_iterations_homo, yhomo*ones(length(good_iterations_homo)), 'ob', plot_props{:});" << std::endl;
  f << "hold on" << std::endl;
  f << "h2 = plot(really_good_iterations_homo, yhomo*ones(length(really_good_iterations_homo)), '*b', plot_props{:});" << std::endl;
  f << "h3 = plot(good_iterations_lumo, ylumo*ones(length(good_iterations_lumo)), 'sr', plot_props{:}); " << std::endl;
  f << "h4 = plot(really_good_iterations_lumo, ylumo*ones(length(really_good_iterations_lumo)), '+r', plot_props{:});" << std::endl;
  f << "grid on" << std::endl;
  f << "set(gca, 'XMinorGrid','off', 'YMinorGrid','off', 'GridAlpha',0.6, 'GridLineStyle', '--')" << std::endl;
  f << "set(gca,'FontSize',20);" << std::endl;
  f << "xlim([0 total_it]);" << std::endl;
  f << "set(gca,'TickLabelInterpreter', 'latex');" << std::endl;
  f << "set(gca,'XTick',[0 5:5:total_it]);" << std::endl;
  f << "set(gca,'YTick',[1 2]);" << std::endl;
  f << "set(gca,'YTickLabel', {'Lumo','Homo'})" << std::endl;
  f << "ylim([0 3]); " << std::endl;
  f << "xlabel('Iteration SP2', 'interpreter','latex');" << std::endl;
  f << "legend([h1(1), h2(1), h3(1), h4(1)], {'Ergo thinks (homo)', 'Actual value (homo)', 'Ergo thinks (lumo)', 'Actual value (lumo)'}, 'interpreter','latex',..." << std::endl;
  f << "    'Orientation', 'Horizontal');" << std::endl;
  f.close();
}


#endif //HEADER_PURIFICATION_GENERAL

